#==============================================================================#
#####     ESTADÍSTICAS DE EDIFICACIONES (PERMISOS DE CONSTRUCCIÓN) 2023     ####
#==============================================================================#
#==============================================================================#
## GENERACIÓN DE TABULADOS - ESTADÍSTICAS DE EDIFICACIONES IV TRIMESTRE 2023 ##
#==============================================================================#
#==============================================================================#
#####           INSTITUTO NACIONAL DE ESTADÍSTICA Y CENSOS (INEC)          ####*
#==============================================================================#
# FUENTE: Estadísticas de Edificaciones 2020 - 2023.
# RESPONSABLE: Gestión de Estadísticas Estructurales (GESE).
# Fecha de elaboración: 25/03/2024.
# Fecha de última actualización: 28/03/2024.
# ==============================================================================*
# ==============================================================================*
# Elaborado por:
# Unidad de Gestión de Estadísticas Estructurales (GESE).
# Valery Paz y Miño
# Enrique Vallejo
# ==============================================================================*
# Revisado por:
# Lorena Ramos
# Responsable de Estadísticas de Edificaciones (ESED)
# Roberto Chaves
# Jefe de Gestión de Estadísticas Estructurales (GESE)
# Instituto Nacional de Estadística y Censos
# ==============================================================================*
# Aprobado por: 
# Diana Barco
# Directora de Estadísticas Económicas (DECON)
# Instituto Nacional de Estadística y Censos 
# Diana_Barco@inec.gob.ec
# ==============================================================================*

# ==============================================================================*
######                          INSTRUCCIONES                             #####*
# ==============================================================================*
# 1.- Descargar la base de datos de las Estadísticas de Edificaciones de 2019, 
#     2020, 2021, 2022 y 2023 (IV trimestre) en formato SPSS de la página web
#     oficial del INEC.
#     (https://www.ecuadorencifras.gob.ec/edificaciones/).

# 2.- En el DISCO C de su computadora, crear una carpeta llamada "Tabulados ESED IVT 2023". 
# En esta carpeta  guardar las bases de datos históricas y la carpeta llamada "Títulos tabulados",
# aquí se exportará todos los resultados en Excel (Tabulados).
# =============================================================================*

####      ************************ NOTA ******************************    #### *
# En este SCRIPT, se encuentra toda la programación necesaria para la generación
# de los diferentes tabulados, mismos que fueron realizados en R-STUDIO 4.2.2. 
# **************************************************************************** #

# ============================================================================ #
######                            PASOS PREVIOS                           ######
# ============================================================================ #

# Limpiar el espacio de trabajo

rm(list=ls())
cat("\014")

#==============================================================================#
#                               PAQUETES                                       #
#==============================================================================#

# De ser necesario instalar cada librería retirando los '#' (luego de ello, colocar
# nuevamente los '#' )

  # install.packages("tidyverse")
  # install.packages("openxlsx")
  # install.packages("dplyr")
  # install.packages("haven")

#==============================================================================#
###                      CARGA DE LIBRERIAS                                ####*
#==============================================================================#

library(tidyverse)
library(openxlsx)
library(dplyr)
library(haven)

#==============================================================================#
###                IMPORTACIÓN BASE DE DATOS Y RUTA DE TRABAJO              ####
#==============================================================================#

setwd("C:/Tabulados ESED IVT 2023") # Directorio de trabajo del IVT trimestre. 
dir()  # Ver los documentos que reposan en el directorio de trabajo establecido. 

#==============================================================================#
#####                    UNIFICACIÓN DE BASES DE DATOS                     #####
#==============================================================================#

# Con las bases descargadas de la pagina oficial del INEC, se procede a la unificación de las 
# bases de datos 

db19 <- read_sav("5. 2019_ENED_BDD.sav") # Base de datos del año 2019
db20 <- read_sav("5. 2020_ENED_BDD.sav") # Base de datos del año 2020
db21 <- read_sav("7. 2021_ESED_BDD.sav") # Base de datos del año 2021
db22 <- read_sav("6. 2022_ESED_BDD.sav") # Base de datos del año 2022
db23_IV <- read_sav("6. 2023_ESED_BDD_IVT.sav") # Base de datos actual - IV trimestre 2023 (Acumulado)

db20 <- db20 %>%
  mutate(anio = case_when(anio == 2021 ~ 2020,
                          TRUE ~ anio))

db19_23 <- bind_rows(db19, db20, db21, db22, db23_IV) # Base general

rm(db19, db20, db21, db22, db23_IV)

#==============================================================================#
#####            BASES DE DATOS FINAL A USAR EN EL SCRIPT                  ####*
#==============================================================================#

ESED_2023_IV <- db19_23 # Base de datos conjunta, mas el IV trimeste (acumulado) del 2023 

dat <- db19_23  # Se utilizará esta BDD para la generación de las nuevas CATEGORIAS   

rm(db19_23) # Se elimina esta BDD, ya que ESED_2023_IV será la base donde se realizará todo el proceso

# ============================================================================ #
#####               CREACIÓN DE VARIABLES PARA LOS TABULADOS               #####
# ============================================================================ #

dat <- dat %>% 
  mutate(t = 1) %>% 
  mutate(tg = 1) 

attr(dat$t, "label") <- "TOTAL"
attr(dat$tg, "label") <- "TOTAL NACIONAL"

ESED_2023_IV$t <- dat$t
ESED_2023_IV$tg <- dat$tg

# ============================================================================ #
#####               RECODIFICACION DE VARIABLES PARA LOS TABULADOS          #####
# ============================================================================ #

## Uso de la edificación (CDISPUSO), cruzada con Propiedad de la edificación (propie)
# para desagregar entre; Educación Pública y Privada, y Hospitales Públicos y Privados

dat <- dat %>% 
  mutate(CDISPUSX = case_when(CDISPUSO == 0 ~ 0,
                              CDISPUSO == 1 ~ 1,
                              CDISPUSO == 2 ~ 2,
                              CDISPUSO == 3 ~ 3,
                              CDISPUSO == 4 ~ 4,
                              CDISPUSO == 5 ~ 5,
                              CDISPUSO == 6 ~ 6,
                              CDISPUSO == 7 & propie == 1 ~ 8,
                              CDISPUSO == 7 & propie == 2 ~ 9,
                              CDISPUSO == 8 ~ 10,
                              CDISPUSO == 9 ~ 11,
                              CDISPUSO == 10 & propie == 1 ~ 13,
                              CDISPUSO == 10 & propie == 2 ~ 14,
                              CDISPUSO == 11 ~ 15,
                              CDISPUSO == 13 ~ 17,
                              CDISPUSO == 14 ~ 18,
                              T ~ 0
                              )) %>%
  mutate(CDISPUSX = factor(CDISPUSX, 
                       levels = c(0, 1, 2, 3, 4, 5, 6, 8, 9,
                                  10, 11, 13, 14, 15, 17, 18),
                       labels = c("No Aplica",
                                  "Edificaciones con una vivienda",
                                  "Edificaciones con dos viviendas",
                                  "Edificaciones con tres o más viviendas",
                                  "Comercial",
                                  "Industrial",
                                  "Edificio Administrativo",
                                  "Educación Privada",
                                  "Educación Pública",
                                  "Cultura",
                                  "Recreación Deporte",
                                  "Hospitales, Clínicas Estab de Salud Privada",
                                  "Hospitales, Clínicas Estab de Salud Pública",
                                  "Transporte y Comunicaciones",
                                  "Mixto Residencial y No Residencial",
                                  "Otras")))

attr(dat$CDISPUSX, "label") <- "PROPÓSITO DE LA CONSTRUCCIÓN"

ESED_2023_IV$CDISPUSX <- dat$CDISPUSX

###### Se crea la variable total de área a construir sin parqueadero ####*

dat$CARCSING <- dat$CARCO - dat$CARDPAR

attr(dat$CARCSING, "label") <- "AREA TOTAL SIN PARQUEADERO"

ESED_2023_IV$CARCSING <- dat$CARCSING

###### Número de pisos para edificaciones de tipo residencial ######*

dat <- dat %>% 
  mutate(TCDISREP = case_when(CDISPUSO >= 1 & CDISPUSO <= 3 ~ 1, T ~ 0)) %>% 
  mutate(CDISREP = case_when(CDISPUSO >= 1 & CDISPUSO <= 3 & NUPICAL == 1 ~ 1,
                             CDISPUSO >= 1 & CDISPUSO <= 3 & NUPICAL == 2 ~ 2,
                             CDISPUSO >= 1 & CDISPUSO <= 3 & NUPICAL == 3 ~ 3,
                             CDISPUSO >= 1 & CDISPUSO <= 3 & NUPICAL >= 4 ~ 4,
                             T ~ 0
                             )) %>% 
  mutate(CDISREP = factor(CDISREP, levels = c(1:4),
                          labels = c("1 PISO",
                                     "2 PISOS",
                                     "3 PISOS",
                                     "4 Y MAS PISOS"
                                     )))

attr(dat$CDISREP, "label") <- "RESIDENCIAL"
attr(dat$TCDISREP, "label") <- "SUBTOTAL RESIDENCIAL" 

ESED_2023_IV$TCDISREP <- dat$TCDISREP
ESED_2023_IV$CDISREP <- dat$CDISREP

###### Número de pisos para edificaciones de tipo No Residencial ######*

dat <- dat %>% 
  mutate(TCDISNRP = case_when(((CDISPUSO >= 4 & CDISPUSO <= 12) | CDISPUSO == 14 ~ 1))) %>% 
  mutate(CDISNRP = case_when((CDISPUSO >= 4 & CDISPUSO <= 12 | CDISPUSO == 14) & NUPICAL == 1 ~ 1,
                             (CDISPUSO >= 4 & CDISPUSO <= 12 | CDISPUSO == 14) & NUPICAL == 2 ~ 2,
                             (CDISPUSO >= 4 & CDISPUSO <= 12 | CDISPUSO == 14) & NUPICAL == 3 ~ 3,
                             (CDISPUSO >= 4 & CDISPUSO <= 12 | CDISPUSO == 14) & NUPICAL >= 4 ~ 4,
                             T ~ 0
                             )) %>% 
  mutate(CDISNRP = factor(CDISNRP, levels = c(1:4),
                           labels = c("1 PISO",
                                      "2 PISOS",
                                      "3 PISOS",
                                      "4 Y MAS PISOS"
                                      )))

attr(dat$TCDISNRP, "label")  <- "SUBTOTAL NO RESIDENCIAL"
attr(dat$CDISNRP, "label")  <- "NO RESIDENCIAL"

ESED_2023_IV$TCDISNRP <- dat$TCDISNRP
ESED_2023_IV$CDISNRP <- dat$CDISNRP

###### Número de pisos para edificaciones de tipo Mixta ######*

dat <- dat %>% 
  mutate(TCDISMIP = ifelse((CDISPUSO == 13), 1, NA)) %>% 
  mutate(CDISMIP = case_when(CDISPUSO == 13 & NUPICAL == 1 ~ 1,
                             CDISPUSO == 13 & NUPICAL == 2 ~ 2,
                             CDISPUSO == 13 & NUPICAL == 3 ~ 3,
                             CDISPUSO == 13 & NUPICAL >= 4 ~ 4
                             )) %>% 
  mutate(CDISMIP = factor(CDISMIP, levels = c(1:4),
                           labels = c("1 PISO",
                                      "2 PISOS",
                                      "3 PISOS",
                                      "4 Y MAS PISOS"
                                      )))

attr(dat$TCDISMIP, "label")  <- "SUBTOTAL MIXTA"
attr(dat$CDISMIP, "label") <- "MIXTA"

ESED_2023_IV$TCDISMIP <- dat$TCDISMIP
ESED_2023_IV$CDISMIP <- dat$CDISMIP

###### Número de cuartos para Residencial de 1 Familia ######*

dat <- dat %>% 
  mutate(TCVIVREP = ifelse((CDISPUSO == 1), 1, NA)) %>% 
  mutate(CVIVREP = case_when(CDISPUSO == 1 & NUCUCAL == 1 ~ 1,
                             CDISPUSO == 1 & NUCUCAL == 2 ~ 2,
                             CDISPUSO == 1 & NUCUCAL == 3 ~ 3,
                             CDISPUSO == 1 & NUCUCAL == 4 ~ 4,
                             CDISPUSO == 1 & NUCUCAL == 5 ~ 5,
                             CDISPUSO == 1 & NUCUCAL == 6 ~ 6,
                             CDISPUSO == 1 & NUCUCAL >= 7 ~ 7
                             )) %>% 
  mutate(CVIVREP = factor(CVIVREP, levels = c(1:7),
                           labels = c("1 CUARTO",
                                      "2 CUARTOS",
                                      "3 CUARTOS",
                                      "4 CUARTOS",
                                      "5 CUARTOS",
                                      "6 CUARTOS",
                                      "7 Y MAS CUARTOS"
                                      )))

attr(dat$TCVIVREP, "label") <- "SUBTOTAL 1 FAMILIA"
attr(dat$CVIVREP, "label") <- "1 FAMILIA"

ESED_2023_IV$TCVIVREP <- dat$TCVIVREP
ESED_2023_IV$CVIVREP <- dat$CVIVREP

###### Número de cuartos para Residencial de 2 Familias ###### *

dat <- dat %>% 
  mutate(TCVIVRE2 = ifelse(CDISPUSO == 2, 1, NA)) %>% 
  mutate(CVIVRE2 = case_when(CDISPUSO == 2 & NUCUCAL == 1 ~ 1,
                             CDISPUSO == 2 & NUCUCAL == 2 ~ 2,
                             CDISPUSO == 2 & NUCUCAL == 3 ~ 3,
                             CDISPUSO == 2 & NUCUCAL == 4 ~ 4,
                             CDISPUSO == 2 & NUCUCAL == 5 ~ 5,
                             CDISPUSO == 2 & NUCUCAL == 6 ~ 6,
                             CDISPUSO == 2 & NUCUCAL >= 7 ~ 7
                             )) %>% 
  mutate(CVIVRE2 = factor(CVIVRE2, levels = c(1:7),
                          labels = c("1 CUARTO",
                                     "2 CUARTOS",
                                     "3 CUARTOS",
                                     "4 CUARTOS",
                                     "5 CUARTOS",
                                     "6 CUARTOS",
                                     "7 Y MAS CUARTOS"
                                     )))

attr(dat$TCVIVRE2, "label") <- "SUBTOTAL 2 FAMILIAS"
attr(dat$CVIVRE2, "label") <- "2 FAMILIAS"

ESED_2023_IV$TCVIVRE2 <- dat$TCVIVRE2
ESED_2023_IV$CVIVRE2 <- dat$CVIVRE2

###### Número de cuartos para Residencial de 3 Familias ###### *

dat <- dat %>% 
  mutate(TCVIVRE3 = ifelse((CDISPUSO == 3),1,NA)) %>% 
  mutate(CVIVRE3 = case_when(CDISPUSO == 3 & NUCUCAL == 1 ~ 1,
                             CDISPUSO == 3 & NUCUCAL == 2 ~ 2,
                             CDISPUSO == 3 & NUCUCAL == 3 ~ 3,
                             CDISPUSO == 3 & NUCUCAL == 4 ~ 4,
                             CDISPUSO == 3 & NUCUCAL == 5 ~ 5,
                             CDISPUSO == 3 & NUCUCAL == 6 ~ 6,
                             CDISPUSO == 3 & NUCUCAL >= 7 ~ 7
                             )) %>% 
  mutate(CVIVRE3 = factor(CVIVRE3, levels = c(1:7),
                          labels = c("1 CUARTO",
                                     "2 CUARTOS",
                                     "3 CUARTOS",
                                     "4 CUARTOS",
                                     "5 CUARTOS",
                                     "6 CUARTOS",
                                     "7 Y MAS CUARTOS"
                                     )))
         
attr(dat$TCVIVRE3, "label") <- "SUBTOTAL 3 FAMILIAS"
attr(dat$CVIVRE3, "label") <- "3 FAMILIAS"

ESED_2023_IV$TCVIVRE3 <- dat$TCVIVRE3
ESED_2023_IV$CVIVRE3 <- dat$CVIVRE3

###### Número de cuartos para Residencial Mixta ######*

dat <- dat %>% 
  mutate(TCVIVRE4 = ifelse((CDISPUSO == 13), 1, NA)) %>% 
  mutate(CVIVRE4 = case_when(CDISPUSO == 13 & NUCUCAL == 1 ~ 1,
                             CDISPUSO == 13 & NUCUCAL == 2 ~ 2,
                             CDISPUSO == 13 & NUCUCAL == 3 ~ 3,
                             CDISPUSO == 13 & NUCUCAL == 4 ~ 4,
                             CDISPUSO == 13 & NUCUCAL == 5 ~ 5,
                             CDISPUSO == 13 & NUCUCAL == 6 ~ 6,
                             CDISPUSO == 13 & NUCUCAL >= 7 ~ 7
                             )) %>% 
  mutate(CVIVRE4 = factor(CVIVRE4, levels = c(1:7),
                          labels = c("1 CUARTO",
                                     "2 CUARTOS",
                                     "3 CUARTOS",
                                     "4 CUARTOS",
                                     "5 CUARTOS",
                                     "6 CUARTOS",
                                     "7 Y MAS CUARTOS"
                                     )))

attr(dat$TCVIVRE4, "label") <- "SUBTOTAL MIXTA"
attr(dat$CVIVRE4, "label") <- "MIXTA"

ESED_2023_IV$TCVIVRE4 <- dat$TCVIVRE4
ESED_2023_IV$CVIVRE4 <- dat$CVIVRE4


###### Número de dormitorios para  Residencial 1 Familia ###### *

dat <- dat %>% 
  mutate(TDVIVREP = ifelse((CDISPUSO == 1), 1, NA)) %>% 
  mutate(DVIVREP = case_when(CDISPUSO == 1 & NUDOCAL == 1 ~ 1,
                             CDISPUSO == 1 & NUDOCAL == 2 ~ 2,
                             CDISPUSO == 1 & NUDOCAL == 3 ~ 3,
                             CDISPUSO == 1 & NUDOCAL == 4 ~ 4,
                             CDISPUSO == 1 & NUDOCAL == 5 ~ 5,
                             CDISPUSO == 1 & NUDOCAL == 6 ~ 6,
                             CDISPUSO == 1 & NUDOCAL >= 7 ~ 7
                             )) %>% 
  mutate(DVIVREP = factor(DVIVREP, levels = c(1:7),
                          labels = c("1 DORMITORIO",
                                     "2 DORMITORIOS",
                                     "3 DORMITORIOS",
                                     "4 DORMITORIOS",
                                     "5 DORMITORIOS",
                                     "6 DORMITORIOS",
                                     "7 Y MAS DORMITORIOS"
                                     )))

attr(dat$TDVIVREP, "label") <- "SUBTOTAL 1 FAMILIA"
attr(dat$DVIVREP, "label") <- "1 FAMILIA"

ESED_2023_IV$TDVIVREP <- dat$TDVIVREP
ESED_2023_IV$DVIVREP <- dat$DVIVREP


###### Número de dormitorios para Residencial 2 Familias ###### *

dat <- dat %>% 
  mutate(TDVIVRE2 = ifelse((CDISPUSO == 2), 1, NA)) %>% 
  mutate(DVIVRE2 = case_when(CDISPUSO == 2 & NUDOCAL == 1 ~ 1,
                             CDISPUSO == 2 & NUDOCAL == 2 ~ 2,
                             CDISPUSO == 2 & NUDOCAL == 3 ~ 3,
                             CDISPUSO == 2 & NUDOCAL == 4 ~ 4,
                             CDISPUSO == 2 & NUDOCAL == 5 ~ 5,
                             CDISPUSO == 2 & NUDOCAL == 6 ~ 6,
                             CDISPUSO == 2 & NUDOCAL >= 7 ~ 7
                             )) %>% 
  mutate(DVIVRE2 = factor(DVIVRE2, levels = c(1:7),
                          labels = c("1 DORMITORIO",
                                     "2 DORMITORIOS",
                                     "3 DORMITORIOS",
                                     "4 DORMITORIOS",
                                     "5 DORMITORIOS",
                                     "6 DORMITORIOS",
                                     "7 Y MAS DORMITORIOS"
                                     )))

attr(dat$TDVIVRE2, "label") <- "SUBTOTAL 2 FAMILIAS"
attr(dat$DVIVRE2, "label") <- "2 FAMILIAS"

ESED_2023_IV$TDVIVRE2 <- dat$TDVIVRE2
ESED_2023_IV$DVIVRE2 <- dat$DVIVRE2


###### Número de dormitorios para Residencial 3 Familias ######*

dat <- dat %>% 
  mutate(TDVIVRE3 = ifelse((CDISPUSO == 3),1,NA)) %>% 
  mutate(DVIVRE3 = case_when(CDISPUSO == 3 & NUDOCAL == 1 ~ 1,
                             CDISPUSO == 3 & NUDOCAL == 2 ~ 2,
                             CDISPUSO == 3 & NUDOCAL == 3 ~ 3,
                             CDISPUSO == 3 & NUDOCAL == 4 ~ 4,
                             CDISPUSO == 3 & NUDOCAL == 5 ~ 5,
                             CDISPUSO == 3 & NUDOCAL == 6 ~ 6,
                             CDISPUSO == 3 & NUDOCAL >= 7 ~ 7
                             )) %>% 
  mutate(DVIVRE3 = factor(DVIVRE3, levels = c(1:7),
                          labels = c("1 DORMITORIO",
                                     "2 DORMITORIOS",
                                     "3 DORMITORIOS",
                                     "4 DORMITORIOS",
                                     "5 DORMITORIOS",
                                     "6 DORMITORIOS",
                                     "7 Y MAS DORMITORIOS"
                                     )))

attr(dat$TDVIVRE3, "label") <- "SUBTOTAL 3 FAMILIAS"
attr(dat$DVIVRE3, "label") <- "3 FAMILIAS"

ESED_2023_IV$TDVIVRE3 <- dat$TDVIVRE3
ESED_2023_IV$DVIVRE3 <- dat$DVIVRE3

###### Número de dormitorios para Residencial Mixta ###### *

dat <- dat %>% 
  mutate(TDVIVRE4 = ifelse((CDISPUSO == 13), 1, NA)) %>% 
  mutate(DVIVRE4 = case_when(CDISPUSO == 13 & NUDOCAL == 1 ~ 1,
                             CDISPUSO == 13 & NUDOCAL == 2 ~ 2,
                             CDISPUSO == 13 & NUDOCAL == 3 ~ 3,
                             CDISPUSO == 13 & NUDOCAL == 4 ~ 4,
                             CDISPUSO == 13 & NUDOCAL == 5 ~ 5,
                             CDISPUSO == 13 & NUDOCAL == 6 ~ 6,
                             CDISPUSO == 13 & NUDOCAL >= 7 ~ 7
                             )) %>% 
  mutate(DVIVRE4 = factor(DVIVRE4, levels = c(1:7),
                          labels = c("1 DORMITORIO",
                                     "2 DORMITORIOS",
                                     "3 DORMITORIOS",
                                     "4 DORMITORIOS",
                                     "5 DORMITORIOS",
                                     "6 DORMITORIOS",
                                     "7 Y MAS DORMITORIOS"
                                     )))

attr(dat$TDVIVRE4, "label") <- "SUBTOTAL MIXTA"
attr(dat$DVIVRE4, "label") <- "MIXTA"

ESED_2023_IV$TDVIVRE4 <- dat$TDVIVRE4
ESED_2023_IV$DVIVRE4 <- dat$DVIVRE4


###### Se recodifica la variable CIMIENTOS para que el "No aplica" se cargue a la categoría "otros" ###*

dat <- dat %>% 
  mutate(RCIMI = case_when(cimi == 120 ~ 120,
                           cimi == 124 ~ 124,
                           cimi == 128 ~ 128,
                           cimi == 132 ~ 132,
                           cimi == 147 ~ 147,
                           cimi == 136 ~ 136,
                           cimi == 0 ~ 136,
                           T ~ 0
                           )) %>% 
  mutate(RCIMI = factor(RCIMI, levels = c(120, 124, 128, 132, 147, 136),
                        labels = c("HORMIGÓN ARMADO",
                                   "HORMIGÓN CICLÓPEO",
                                   "PILOTES MADERA",
                                   "PILOTES HORMIGÓN",
                                   "CIMIENTOS PORTANTES",
                                   "OTROS"
                                   )))

attr(dat$RCIMI, "label") <- "CIMIENTOS"

ESED_2023_IV$RCIMI <- dat$RCIMI


###### Se recodifica la variable ESTRUCTURA para que el "No aplica" se cargue a la categoría "otros" ##*

dat <- dat %>% 
  mutate(RESTRU = case_when(estru == 121 ~ 121,
                            estru == 125 ~ 125,
                            estru == 129 ~ 129,
                            estru == 148 ~ 148,
                            estru == 133 ~ 133,
                            estru == 0 ~ 133,
                            T ~ 0
                            )) %>% 
  mutate(RESTRU = factor(RESTRU, levels = c(121, 125, 129, 148, 133),
                         labels = c("HORMIGÓN ARMADO",
                                    "METÁLICA",
                                    "MADERA",
                                    "MUROS PORTANTES",
                                    "OTROS"
                                    )))

attr(dat$RESTRU, "label") <- "ESTRUCTURA"

ESED_2023_IV$RESTRU <- dat$RESTRU


###### Se recodifica la variable PARED para que el "No aplica" se cargue a la categoría "otros" #*

dat <- dat %>% 
  mutate(RPARED = case_when(pared == 122 ~ 122,
                            pared == 126 ~ 126,
                            pared == 130 ~ 130,
                            pared == 134 ~ 134,
                            pared == 138 ~ 138,
                            pared == 142 ~ 142,
                            pared == 149 ~ 149,
                            pared == 150 ~ 150,
                            pared == 146 ~ 146,
                            pared == 0 ~ 146,
                            T ~ 0
                            )) %>% 
  mutate(RPARED = factor(RPARED, levels = c(122, 126, 130, 134, 138,
                                            142, 149, 150, 146),
                         labels = c("LADRILLO",
                                    "BLOQUE",
                                    "MADERA",
                                    "ADOBE TAPIA",
                                    "CAÑA REVESTIDA",
                                    "PREFABRICADA",
                                    "PAREDES PORTANTES",
                                    "LAMINAS DE YESO-CARTÓN",
                                    "OTROS"
                                    )))
attr(dat$RPARED, "label") <- "PARED"

ESED_2023_IV$RPARED <- dat$RPARED


###### Se recodifica la variable CUBIERTA para que el "No aplica" se cargue a la categoría "otros" #*

dat <- dat %>% 
  mutate(RCUBI = case_when(cubi == 123 ~ 123,
                           cubi == 127 ~ 127,
                           cubi == 131 ~ 131,
                           cubi == 135 ~ 135,
                           cubi == 139 ~ 139,
                           cubi == 143 ~ 143,
                           cubi == 0 ~ 143,
                           T ~ 0
                           )) %>% 
  mutate(RCUBI = factor(RCUBI, levels = c(123, 127, 131, 135, 139, 143),
                        labels = c("HORMIGÓN ARMADO",
                                   "ETERNIT ASBESTO",
                                   "TEJA",
                                   "ZINC",
                                   "POLICARBONATO",
                                   "OTROS"
                                   )))
attr(dat$RCUBI, "label") <- "CUBIERTA"

ESED_2023_IV$RCUBI <- dat$RCUBI


###### Se recodifica la variable PISOS para que el "No aplica" se cargue a la categoría "otros" #*

dat <- dat %>% 
  mutate(RPISO = case_when(piso == 151 ~ 151,
                           piso == 152 ~ 152,
                           piso == 153 ~ 153,
                           piso == 154 ~ 154,
                           piso == 155 ~ 155,
                           piso == 156 ~ 156,
                           piso == 157 ~ 157,
                           piso == 0 ~ 157,
                           T ~ 0
                           )) %>% 
  mutate(RPISO = factor(RPISO, levels = c(151:157),
                        labels = c("TIERRA ESTABILIZADA",
                                   "HORMIGÓN",
                                   "MADERA",
                                   "CERÁMICA",
                                   "PORCELANATO",
                                   "PISO LAMINADO(PISO FLOTANTE)",
                                   "OTROS"
                                   )))

attr(dat$RPISO, "label") <- "PISOS"

ESED_2023_IV$RPISO <- dat$RPISO


#### Para los tabulados se debe crear la variable NÚMERO DE PERMISOS DE CONSTRUCCIÓN,
#### que es igual a la variable CPERM, pero en ves de sumar se cuente. ###### *

dat <- dat %>% 
  mutate(NUMPER = CPERM)

attr(dat$NUMPER, "label") <- "NÚMERO DE PERMISOS DE CONSTRUCCIÓN"

ESED_2023_IV$NUMPER <- dat$NUMPER


#### Debemos generar un nueva variable: NÚMERO DE PISOS TOTAL, ya que está por 
### uso residencial, no residencial y Mixta ##*

dat$NUPISOS <- ifelse(!is.na(dat$CDISREP), dat$CDISREP,
                      ifelse(!is.na(dat$CDISNRP), dat$CDISNRP,
                             ifelse(!is.na(dat$CDISMIP), dat$CDISMIP, NA
                                    )))

attr(dat$NUPISOS, "label") <- "NUMERO DE PISOS GENERAL"

ESED_2023_IV$NUPISOS <- dat$NUPISOS


#### Debemos generar una variable: TOTAL DEL NUMERO DE CUARTOS, ya que está desagregada por 
#### Edificaciones de 1 vivienda, Edificaciones de 2 viviendas, Edificaciones de 3 viviendas y mixtas*

dat$NUCUARTOS <- ifelse(!is.na(dat$CVIVREP), dat$CVIVREP,
                        ifelse(!is.na(dat$CVIVRE2), dat$CVIVRE2,
                               ifelse(!is.na(dat$CVIVRE3), dat$CVIVRE3,
                                      ifelse(!is.na(dat$CVIVRE4), dat$CVIVRE4, NA
                                             ))))

attr(dat$NUCUARTOS, "label") <- "NUMERO DE CUARTOS GENERAL"

ESED_2023_IV$NUCUARTOS <- dat$NUCUARTOS


#### Debemos generar una variable TOTAL DEL NUMERO DE DORMITORIOS, ya que está desagregada 
## por Edificaciones de 1 vivienda, Edificaciones de 2 viviendas, Edificaciones de 3 viviendas y mixtas*

dat$NUDORMITORIOS <- ifelse(!is.na(dat$DVIVREP), dat$DVIVREP,
                            ifelse(!is.na(dat$DVIVRE2), dat$DVIVRE2,
                                   ifelse(!is.na(dat$DVIVRE3), dat$DVIVRE3,
                                          ifelse(!is.na(dat$DVIVRE4), dat$DVIVRE4, NA
                                                 ))))
                      
attr(dat$NUDORMITORIOS, "label") <- "NUMERO DE DORMITORIOS GENERAL"

ESED_2023_IV$NUDORMITORIOS <- dat$NUDORMITORIOS

##### Se debe generar una variable FINAN que separe a los recursos propios y prestados ####* 

dat <- dat %>% 
  mutate(FINAN = case_when((CDISPRFI == 1 |  CDISPRFI == 2 |  CDISPRFI == 3 | 
                              CDISPRFI == 4 |  CDISPRFI == 5) ~ 1,
                           (CDISPRFI == 6 |  CDISPRFI == 7 |  CDISPRFI == 8 |  
                              CDISPRFI == 9 |  CDISPRFI == 10 |  CDISPRFI == 11 | 
                              CDISPRFI == 12 | CDISPRFI == 13) ~ 2,
                           T ~ 0
                           )) %>% 
  mutate(FINAN = factor(FINAN, levels = c(0, 1, 2),
                        labels = c("No Aplica",
                                   "Recursos Propios",
                                   "Recursos Prestados"
                                   )))
attr(dat$FINAN, "label") <- "FINANCIAMIENTO PROPIO O PRESTADO"

ESED_2023_IV$FINAN <- dat$FINAN

#### Se crea una variable llamada "TRIMES"  para optener los trimestres por cada año (2020-2023) #*

dat <- dat %>% 
  mutate(TRIMES = case_when(anio == 2019 & (mes == 1 |  mes == 2 |  mes == 3) ~ 1,
                            anio == 2019 & (mes == 4 |  mes == 5 |  mes == 6) ~ 2,
                            anio == 2019 & (mes == 7 |  mes == 8 |  mes == 9) ~ 3,
                            anio == 2019 & (mes == 10 |  mes == 11 |  mes == 12) ~ 4,
                            anio == 2020 & (mes == 1 |  mes == 2 | mes == 3) ~ 5,
                            anio == 2020 & (mes == 4 |  mes == 5 | mes == 6) ~ 6,
                            anio == 2020 & (mes == 7 |  mes == 8 | mes == 9) ~ 7,
                            anio == 2020 & (mes == 10 |  mes == 11 | mes == 12) ~ 8,
                            anio == 2021 & (mes == 1 |  mes == 2 | mes == 3) ~ 9,
                            anio == 2021 & (mes == 4 |  mes == 5 | mes == 6) ~ 10,
                            anio == 2021 & (mes == 7 |  mes == 8 | mes == 9) ~ 11,
                            anio == 2021 & (mes == 10 |  mes == 11 | mes == 12) ~ 12,
                            anio == 2022 & (mes == 1 | mes == 2 | mes == 3) ~ 13,
                            anio == 2022 & (mes == 4 | mes == 5 | mes == 6) ~ 14,
                            anio == 2022 & (mes == 7 |  mes == 8 | mes == 9) ~ 15,
                            anio == 2022 & (mes == 10 |  mes == 11 | mes == 12) ~ 16,
                            anio == 2023 & (mes == 1 | mes == 2 | mes == 3) ~ 17,
                            anio == 2023 & (mes == 4 | mes == 5 | mes == 6) ~ 18,
                            anio == 2023 & (mes == 7 | mes == 8 | mes == 9) ~ 19,
                            anio == 2023 & (mes == 10 | mes == 11 | mes == 12) ~ 20,
                            T ~ 0
                            )) %>% 
  mutate(TRIMES = factor(TRIMES, levels = c(0:20),
                         labels = c("No Aplica",
                                    "2019-I",
                                    "2019-II",
                                    "2019-III",
                                    "2019-IV",
                                    "2020-I",
                                    "2020-II",
                                    "2020-III",
                                    "2020-IV",
                                    "2021-I",
                                    "2021-II",
                                    "2021-III",
                                    "2021-IV",
                                    "2022-I",
                                    "2022-II",
                                    "2022-III",
                                    "2022-IV", 
                                    "2023-I",
                                    "2023-II",
                                    "2023-III",
                                    "2023-IV")))

attr(dat$TRIMES, "label") <- "TRIMESTRES POR AÑOS 2019 AL 2023"

ESED_2023_IV$TRIMES <- dat$TRIMES


table(dat$TRIMES)

#### Se filtra por los 15 GAD municipales seleccionados #####*

ESED_2023_IV <- ESED_2023_IV %>% 
  filter(  codcantf == "1701" | codcantf == "0906" | codcantf == "0901"  | 
           codcantf == "1801" | codcantf == "0101" | codcantf == "1001"  | 
           codcantf == "1101" | codcantf == "0601" | codcantf  == "2301" |
           codcantf == "1301" | codcantf == "0916" | codcantf == "1308"  |
           codcantf == "0701" | codcantf == "0907" | codcantf == "2402"  )

# ============================================================================ #
#####                TABULADOS DE LOS PRINCIPALES RESULTADOS               #####
# ============================================================================ #

# Se define la coma "," como separador de decimales
options(OutDec = ",")

###### C-A Cuadro Resumen ###### *

####### C-A1 Resultados de las principales variables #######

# 2019
CA1_2019 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2019-I", 
                       "2019-II", 
                       "2019-III", 
                       "2019-IV")) %>% 
  group_by(TRIMES) %>% 
  summarise(num_per = sum(tg, na.rm = T),
            num_ed = sum(CPERM, na.rm = T),
            num_viv = sum(CNUVICAL, na.rm = T),
            sup = sum(CSUTE, na.rm = T),
            area = sum(CARCO, na.rm = T),
            valor = sum(CVAE, na.rm = T))%>%
  mutate_at(c(2:7), ~replace(., is.na(.), 0))%>%
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CA1_2019)[5] <- "2019"


# 2020
CA1_2020 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2020-I", 
                       "2020-II", 
                       "2020-III", 
                       "2020-IV")) %>% 
  group_by(TRIMES) %>% 
  summarise(num_per = sum(tg, na.rm = T),
            num_ed = sum(CPERM, na.rm = T),
            num_viv = sum(CNUVICAL, na.rm = T),
            sup = sum(CSUTE, na.rm = T),
            area = sum(CARCO, na.rm = T),
            valor = sum(CVAE, na.rm = T))%>%
  mutate_at(c(2:7), ~replace(., is.na(.), 0))%>%
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CA1_2020)[5] <- "2020"

# 2021
CA1_2021 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2021-I", 
                       "2021-II", 
                       "2021-III", 
                       "2021-IV")) %>% 
  group_by(TRIMES) %>% 
  summarise(num_per = sum(tg, na.rm = T),
            num_ed = sum(CPERM, na.rm = T),
            num_viv = sum(CNUVICAL, na.rm = T),
            sup = sum(CSUTE, na.rm = T),
            area = sum(CARCO, na.rm = T),
            valor = sum(CVAE, na.rm = T))%>%
  mutate_at(c(2:7), ~replace(., is.na(.), 0))%>%
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CA1_2021)[5] <- "2021"

# 2022
CA1_2022 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2022-I", 
                       "2022-II", 
                       "2022-III", 
                       "2022-IV")) %>% 
  group_by(TRIMES) %>% 
  summarise(num_per = sum(tg, na.rm = T),
            num_ed = sum(CPERM, na.rm = T),
            num_viv = sum(CNUVICAL, na.rm = T),
            sup = sum(CSUTE, na.rm = T),
            area = sum(CARCO, na.rm = T),
            valor = sum(CVAE, na.rm = T))%>%
  mutate_at(c(2:7), ~replace(., is.na(.), 0)) %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CA1_2022)[5] <- "2022"

# 2023
CA1_2023 <- ESED_2023_IV %>%
  filter(TRIMES %in% c("2023-I" , 
                       "2023-II",
                       "2023-III",
                       "2023-IV")) %>% 
  group_by(TRIMES) %>% 
  summarise(num_per = sum(tg, na.rm = T),
            num_ed = sum(CPERM, na.rm = T),
            num_viv = sum(CNUVICAL, na.rm = T),
            sup = sum(CSUTE, na.rm = T),
            area = sum(CARCO, na.rm = T),
            valor = sum(CVAE, na.rm = T)) %>%
  mutate_at(c(2:7), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CA1_2023)[5] <- "2023"

ESED_2023_IV$CDISPUSO

# Completo
CA1 <- bind_rows(CA1_2020, CA1_2021, CA1_2022, CA1_2023)
names(CA1) <- c("Número de permisos de construcción",
                 "Número de edificaciones",
                 "Número de viviendas",
                 "Superficie del terreno (m2)",
                 "Área total a construir (m2)",
                 "Valor estimado de la edificación (dólares)")

rm(CA1_2019, CA1_2020, CA1_2021, CA1_2022, CA1_2023)

###### C-B Número de permisos de construcción ######*

##### C-B1 Permisos de construcción, por trimestres, según cantones ####

# 2019
CB1_2019 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2019-I", 
                       "2019-II", 
                       "2019-III", 
                       "2019-IV")) %>% 
  select(TRIMES, codcantf, tg) %>%
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(tg, na.rm = T)) %>%
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>%
  rename("Quito" = "Distrito Metropolitano De Quito",
         "Samborondón" = "Samborondon",
         "Durán" = "Duran") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CB1_2019)[5] <- "2019"

# 2020
CB1_2020 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2020-I", 
                       "2020-II", 
                       "2020-III", 
                       "2020-IV")) %>% 
  select(TRIMES, codcantf, tg) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(tg, na.rm = T)) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>% 
  rename("Quito" = "Distrito Metropolitano De Quito",
         "Samborondón" = "Samborondon",
         "Durán" = "Duran") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CB1_2020)[5] <- "2020"

# 2021
CB1_2021 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2021-I", 
                       "2021-II", 
                       "2021-III", 
                       "2021-IV")) %>% 
  select(TRIMES, codcantf, tg) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(tg, na.rm = T)) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>% 
  rename("Quito" = "Distrito Metropolitano De Quito",
         "Samborondón" = "Samborondon",
         "Durán" = "Duran") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CB1_2021)[5] <- "2021"

# 2022
CB1_2022 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2022-I", 
                       "2022-II", 
                       "2022-III", 
                       "2022-IV")) %>% 
  select(TRIMES, codcantf, tg) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(tg, na.rm = T)) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>% 
  rename("Quito" = "Distrito Metropolitano De Quito",
         "Samborondón" = "Samborondon",
         "Durán" = "Duran") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CB1_2022)[5] <- "2022"

# 2023
CB1_2023 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2023-I" ,
                       "2023-II",
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, codcantf, tg) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(tg, na.rm = T)) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>%
  rename("Quito" = "Distrito Metropolitano De Quito",
         "Samborondón" = "Samborondon",
         "Durán" = "Duran") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup()%>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CB1_2023)[5] <- "2023"

# Completo
CB1 <- bind_rows(CB1_2020, CB1_2021, CB1_2022, CB1_2023)

rm(CB1_2019, CB1_2020, CB1_2021, CB1_2022, CB1_2023)


###### C-C Número de edificaciones a construir ######*

####### C-C1 Edificacioens a construir, por trimestres, según cantones #######

# 2019
CC1_2019 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2019-I", 
                       "2019-II", 
                       "2019-III", 
                       "2019-IV")) %>% 
  select(TRIMES, codcantf, CPERM) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>% 
  rename("Quito" = "Distrito Metropolitano De Quito",
         "Samborondón" = "Samborondon",
         "Durán" = "Duran") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CC1_2019)[5] <- "2019"

# 2020
CC1_2020 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2020-I", 
                       "2020-II", 
                       "2020-III", 
                       "2020-IV")) %>% 
  select(TRIMES, codcantf, CPERM) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>% 
  rename("Quito" = "Distrito Metropolitano De Quito",
         "Samborondón" = "Samborondon",
         "Durán" = "Duran") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CC1_2020)[5] <- "2020"

# 2021
CC1_2021 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2021-I", 
                       "2021-II", 
                       "2021-III", 
                       "2021-IV")) %>% 
  select(TRIMES, codcantf, CPERM) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>% 
  rename("Quito" = "Distrito Metropolitano De Quito",
         "Samborondón" = "Samborondon",
         "Durán" = "Duran") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CC1_2021)[5] <- "2021"

# 2022
CC1_2022 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2022-I", 
                       "2022-II", 
                       "2022-III", 
                       "2022-IV")) %>% 
  select(TRIMES, codcantf, CPERM) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>% 
  rename("Quito" = "Distrito Metropolitano De Quito",
         "Samborondón" = "Samborondon",
         "Durán" = "Duran") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CC1_2022)[5] <- "2022"

# 2023
CC1_2023 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2023-I",
                       "2023-II", 
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, codcantf, CPERM) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>% 
  rename("Quito" = "Distrito Metropolitano De Quito",
         "Samborondón" = "Samborondon",
         "Durán" = "Duran") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup()%>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CC1_2023)[5] <- "2023"

# Completo
CC1 <- bind_rows(CC1_2020, CC1_2021, CC1_2022, CC1_2023) 

rm(CC1_2019, CC1_2020, CC1_2021, CC1_2022, CC1_2023)

####### C-C2 Edificaciones a construir, por trimestres, según tipo de obra #######

# 2019
CC2_2019 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2019-I", 
                       "2019-II", 
                       "2019-III", 
                       "2019-IV")) %>% 
  select(TRIMES, CTIPOBR, CPERM) %>% 
  group_by(TRIMES, as_factor(CTIPOBR)) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de edificaciones" = sbtotal) %>% 
  pivot_wider(names_from = `as_factor(CTIPOBR)`, values_from = total) %>% 
  mutate_at(c(2:5), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CC2_2019)[5] <- "2019"  

# 2020
CC2_2020 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2020-I", 
                       "2020-II", 
                       "2020-III", 
                       "2020-IV")) %>% 
  select(TRIMES, CTIPOBR, CPERM) %>% 
  group_by(TRIMES, as_factor(CTIPOBR)) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de edificaciones" = sbtotal) %>% 
  pivot_wider(names_from = `as_factor(CTIPOBR)`, values_from = total) %>% 
  mutate_at(c(2:5), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CC2_2020)[5] <- "2020"

# 2021
CC2_2021 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2021-I", 
                       "2021-II", 
                       "2021-III", 
                       "2021-IV")) %>% 
  select(TRIMES, CTIPOBR, CPERM) %>% 
  group_by(TRIMES, as_factor(CTIPOBR)) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de edificaciones" = sbtotal) %>% 
  pivot_wider(names_from = `as_factor(CTIPOBR)`, values_from = total) %>% 
  mutate_at(c(2:5), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CC2_2021)[5] <- "2021"

# 2022
CC2_2022 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2022-I", 
                       "2022-II", 
                       "2022-III", 
                       "2022-IV")) %>% 
  select(TRIMES, CTIPOBR, CPERM) %>% 
  group_by(TRIMES, as_factor(CTIPOBR)) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de edificaciones" = sbtotal) %>% 
  pivot_wider(names_from = `as_factor(CTIPOBR)`, values_from = total) %>% 
  mutate_at(c(2:5), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CC2_2022)[5] <- "2022"

# 2023
CC2_2023 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2023-I",
                       "2023-II", 
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, CTIPOBR, CPERM) %>% 
  group_by(TRIMES, as_factor(CTIPOBR)) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de edificaciones" = sbtotal) %>% 
  pivot_wider(names_from = `as_factor(CTIPOBR)`, values_from = total) %>% 
  mutate_at(c(2:5), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup()%>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CC2_2023)[5] <- "2023"

# Completo
CC2 <- bind_rows(CC2_2020, CC2_2021, CC2_2022, CC2_2023)

rm(CC2_2019, CC2_2020, CC2_2021, CC2_2022, CC2_2023)


####### C-C3 Edificaciones a construir, por trimestres, según uso de la edificación #######

# 2019
CC3_2019 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2019-I", 
                       "2019-II", 
                       "2019-III", 
                       "2019-IV")) %>% 
  select(TRIMES, CDISPUSX, CPERM) %>% 
  group_by(TRIMES, CDISPUSX) %>% 
  summarise(total = sum(CPERM)) %>%
  complete(CDISPUSX,  fill = list(total = 0)) %>% 
  pivot_wider(names_from = CDISPUSX, values_from = total) %>% 
  mutate_at(c(2:17), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CC3_2019)[5] <- "2019"

# 2020
CC3_2020 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2020-I", 
                       "2020-II", 
                       "2020-III", 
                       "2020-IV")) %>% 
  select(TRIMES, CDISPUSX, CPERM) %>% 
  group_by(TRIMES, CDISPUSX) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>%   
  complete(CDISPUSX, fill = list(total = 0)) %>% 
  pivot_wider(names_from = CDISPUSX, values_from = total) %>% 
  mutate_at(c(2:17), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CC3_2020)[5] <- "2020"

# 2021
CC3_2021 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2021-I", 
                       "2021-II", 
                       "2021-III", 
                       "2021-IV")) %>% 
  select(TRIMES, CDISPUSX, CPERM) %>% 
  group_by(TRIMES, CDISPUSX) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>%   
  complete(CDISPUSX, fill = list(total = 0)) %>% 
  pivot_wider(names_from = CDISPUSX, values_from = total) %>% 
  mutate_at(c(2:17), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CC3_2021)[5] <- "2021"

# 2022
CC3_2022 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2022-I", 
                       "2022-II", 
                       "2022-III", 
                       "2022-IV")) %>% 
  select(TRIMES, CDISPUSX, CPERM) %>% 
  group_by(TRIMES, CDISPUSX) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>%   
  complete(CDISPUSX, fill = list(total = 0)) %>% 
  pivot_wider(names_from = CDISPUSX, values_from = total) %>% 
  mutate_at(c(2:17), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CC3_2022)[5] <- "2022"

# 2023
CC3_2023 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2023-I",
                       "2023-II", 
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, CDISPUSX, CPERM) %>% 
  group_by(TRIMES, CDISPUSX) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>%   
  complete(CDISPUSX, fill = list(total = 0)) %>% 
  pivot_wider(names_from = CDISPUSX, values_from = total) %>% 
  mutate_at(c(2:17), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CC3_2023)[5] <- "2023"

# Completo
CC3 <- bind_rows(CC3_2020, CC3_2021, CC3_2022, CC3_2023) %>% 
  select(- `No Aplica`) %>%
  mutate_at(c(1:15), ~replace(.,is.na(.),0))
names(CC3) <- c("Edificaciones con una vivienda",
                "Edificaciones con dos viviendas",
                "Edificaciones con tres o más viviendas",
                "Comercial",
                "Industrial",
                "Edificio administrativo (público)",
                "Educación particular",
                "Educación pública",
                "Cultura",
                "Complejos recreacionales",
                "Hospitales, clínicas y otros establecimientos de salud particular",
                "Hospitales, clínicas y otros establecimientos de salud pública",
                "Transporte y comunicaciones",
                "Mixto residencial y no residencial",
                "Otras")

rm(CC3_2019, CC3_2020, CC3_2021, CC3_2022, CC3_2023)

####### C-C4 Edificaciones a construir, por trimestres, según acceso a internet/celular y servicios básicos #######

# 2019
CC4_2019_1 <- ESED_2023_IV %>%
  filter(TRIMES %in% c("2019-I",
                       "2019-II",
                       "2019-III",
                       "2019-IV")) %>% 
  select(TRIMES) %>%
  mutate(int_ur1 = 0,
         int_ur2 = 0) %>% 
  group_by(TRIMES) %>% 
  summarise(int_ur1 = sum(int_ur1), 
            int_ur2 = sum(int_ur2)) %>%
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
names(CC4_2019_1) <- c("Ic urbana Acceso",
                       "Ic urbana Sin acceso")

CC4_2019_2 <- ESED_2023_IV %>%
  filter(TRIMES %in% c("2019-I",
                       "2019-II",
                       "2019-III",
                       "2019-IV")) %>% 
  select(TRIMES) %>%
  mutate(int_ru1 = 0,
         int_ru2 = 0) %>% 
  group_by(TRIMES) %>% 
  summarise(int_ru1 = sum(int_ru1), 
            int_ru2 = sum(int_ru2)) %>%
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
names(CC4_2019_2) <- c("Ic rural Acceso",
                       "Ic rural Sin acceso")


CC4_2019 <- bind_cols(CC4_2019_1, CC4_2019_2)
rownames(CC4_2019)[5] <- "2019"

rm(CC4_2019_1, CC4_2019_2, CC4_2019_3, CC4_2019_4)


# 2020
CC4_2020_1 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2020-I",
                       "2020-II",
                       "2020-III",
                       "2020-IV")) %>% 
  select(TRIMES, CPERM, CINTERN, careaur, CCEL) %>% 
  mutate(int_ur = case_when(careaur == 1 & CINTERN == 1  ~ 1,
                            careaur == 1 & CINTERN == 2  ~ 2,
                            T ~ 0)) %>%
  group_by(TRIMES, int_ur) %>% 
  summarise(total = sum(CPERM)) %>% 
  pivot_wider(names_from = int_ur, values_from = total) %>% 
  mutate_at(c(2:4), ~replace(., is.na(.), 0)) %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
names(CC4_2020_1) <- c("Ic urbana no aplica",
                       "Ic urbana Acceso",
                       "Ic urbana Sin acceso")

CC4_2020_2 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2020-I",
                       "2020-II",
                       "2020-III",
                       "2020-IV")) %>% 
  select(TRIMES, CPERM, CINTERN, careaur, CCEL) %>% 
  mutate(int_ru = case_when(careaur == 2 & CINTERN == 1  ~ 1,
                            careaur == 2 & CINTERN == 2  ~ 2,
                            T ~ 0)) %>%
  group_by(TRIMES, int_ru) %>% 
  summarise(total = sum(CPERM)) %>% 
  pivot_wider(names_from = int_ru, values_from = total) %>% 
  mutate_at(c(2:4), ~replace(., is.na(.), 0)) %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
names(CC4_2020_2) <- c("Ic rural no aplica",
                       "Ic rural Acceso",
                       "Ic rural Sin acceso")

CC4_2020 <- bind_cols(CC4_2020_1, CC4_2020_2) %>% 
  select(-"Ic urbana no aplica", -"Ic rural no aplica")
rownames(CC4_2020)[5] <- "2020"

rm(CC4_2020_1, CC4_2020_2)


# 2021
CC4_2021_1 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2021-I",
                       "2021-II",
                       "2021-III",
                       "2021-IV")) %>% 
  select(TRIMES, CPERM, CINTERN, careaur, CCEL) %>% 
  mutate(int_ur = case_when(careaur == 1 & CINTERN == 1  ~ 1,
                            careaur == 1 & CINTERN == 2  ~ 2,
                            T ~ 0)) %>%
  group_by(TRIMES, int_ur) %>% 
  summarise(total = sum(CPERM)) %>% 
  pivot_wider(names_from = int_ur, values_from = total) %>% 
  mutate_at(c(2:4), ~replace(., is.na(.), 0)) %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
names(CC4_2021_1) <- c("Ic urbana no aplica",
                       "Ic urbana Acceso",
                       "Ic urbana Sin acceso")

CC4_2021_2 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2021-I",
                       "2021-II",
                       "2021-III",
                       "2021-IV")) %>% 
  select(TRIMES, CPERM, CINTERN, careaur, CCEL) %>% 
  mutate(int_ru = case_when(careaur == 2 & CINTERN == 1  ~ 1,
                            careaur == 2 & CINTERN == 2  ~ 2,
                            T ~ 0)) %>%
  group_by(TRIMES, int_ru) %>% 
  summarise(total = sum(CPERM)) %>% 
  pivot_wider(names_from = int_ru, values_from = total) %>% 
  mutate_at(c(2:4), ~replace(., is.na(.), 0)) %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
names(CC4_2021_2) <- c("Ic rural no aplica",
                       "Ic rural Acceso",
                       "Ic rural Sin acceso")

CC4_2021 <- bind_cols(CC4_2021_1, CC4_2021_2) %>% 
  select(-"Ic urbana no aplica", -"Ic rural no aplica")
rownames(CC4_2021)[5] <- "2021"

rm(CC4_2021_1, CC4_2021_2)


# 2022
CC4_2022_1 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2022-I",
                       "2022-II",
                       "2022-III",
                       "2022-IV")) %>% 
  select(TRIMES, CPERM, CINTERN, careaur, CCEL) %>% 
  mutate(int_ur = case_when(careaur == 1 & CINTERN == 1  ~ 1,
                            careaur == 1 & CINTERN == 2  ~ 2,
                            T ~ 0)) %>%
  group_by(TRIMES, int_ur) %>% 
  summarise(total = sum(CPERM)) %>% 
  pivot_wider(names_from = int_ur, values_from = total) %>% 
  mutate_at(c(2:4), ~replace(., is.na(.), 0)) %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
names(CC4_2022_1) <- c("Ic urbana no aplica",
                       "Ic urbana Acceso",
                       "Ic urbana Sin acceso")

CC4_2022_2 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2022-I",
                       "2022-II",
                       "2022-III",
                       "2022-IV")) %>% 
  select(TRIMES, CPERM, CINTERN, careaur, CCEL) %>% 
  mutate(int_ru = case_when(careaur == 2 & CINTERN == 1  ~ 1,
                            careaur == 2 & CINTERN == 2  ~ 2,
                            T ~ 0)) %>%
  group_by(TRIMES, int_ru) %>% 
  summarise(total = sum(CPERM)) %>% 
  pivot_wider(names_from = int_ru, values_from = total) %>% 
  mutate_at(c(2:4), ~replace(., is.na(.), 0)) %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
names(CC4_2022_2) <- c("Ic rural no aplica",
                       "Ic rural Acceso",
                       "Ic rural Sin acceso")


CC4_2022 <- bind_cols(CC4_2022_1, CC4_2022_2) %>% 
  select(-"Ic urbana no aplica", -"Ic rural no aplica")
rownames(CC4_2022)[5] <- "2022"

rm(CC4_2022_1, CC4_2022_2)


# 2023
CC4_2023_1 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2023-I", 
                       "2023-II",
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, CPERM, CINTERN, careaur) %>% 
  mutate(int_ur = case_when(careaur == 1 & CINTERN == 1 ~ 1, 
                            careaur == 1 & CINTERN == 2 ~ 2,
                            T ~ 0)) %>%
  group_by(TRIMES, int_ur) %>% 
  summarise(total = sum(CPERM)) %>% 
  pivot_wider(names_from = int_ur, values_from = total) %>% 
  mutate_at(c(2:4), ~replace(., is.na(.), 0)) %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
names(CC4_2023_1) <- c("Ic urbana no aplica",
                       "Ic urbana Acceso",
                       "Ic urbana Sin acceso")

CC4_2023_2 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2023-I",
                       "2023-II", 
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, CPERM, CINTERN, careaur) %>% 
  mutate(int_ru = case_when(careaur == 2 & CINTERN == 1 ~ 1, 
                            careaur == 2 & CINTERN == 2 ~ 2,
                            T ~ 0)) %>%
  group_by(TRIMES, int_ru) %>% 
  summarise(total = sum(CPERM)) %>% 
  pivot_wider(names_from = int_ru, values_from = total) %>% 
  mutate_at(c(2:4), ~replace(., is.na(.), 0)) %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
names(CC4_2023_2) <- c("Ic rural no aplica",
                       "Ic rural Acceso",
                       "Ic rural Sin acceso")

CC4_2023 <- bind_cols(CC4_2023_1, CC4_2023_2) %>% 
  select(-"Ic urbana no aplica", - "Ic rural no aplica")
rownames(CC4_2023)[5] <- "2023"


rm(CC4_2023_1, CC4_2023_2)


# Completo 
CC4 <- bind_rows(CC4_2020, CC4_2021, CC4_2022, CC4_2023)

rm(CC4_2019, CC4_2020, CC4_2021, CC4_2022, CC4_2023)


###### C-D Número de viviendas proyectadas ######*

####### C-D1 Viviendas a construir, por trimestres, según cantones #######

# 2019
CD1_2019 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2019-I", 
                       "2019-II", 
                       "2019-III", 
                       "2019-IV")) %>% 
  select(TRIMES, CNUVICAL, codcantf) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>%
  summarise(total = sum(CNUVICAL)) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:15), ~replace(., is.na(.), 0)) %>%
  rename("Quito" = "Distrito Metropolitano De Quito",
         "Samborondón" = "Samborondon",
         "Durán" = "Duran") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CD1_2019)[5] <- "2019"  

# 2020
CD1_2020 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2020-I", 
                       "2020-II", 
                       "2020-III", 
                       "2020-IV")) %>% 
  select(TRIMES, CNUVICAL, codcantf) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>%
  summarise(total = sum(CNUVICAL)) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:15), ~replace(., is.na(.), 0)) %>%
  rename("Quito" = "Distrito Metropolitano De Quito",
         "Samborondón" = "Samborondon",
         "Durán" = "Duran") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CD1_2020)[5] <- "2020"

# 2021
CD1_2021 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2021-I", 
                       "2021-II", 
                       "2021-III", 
                       "2021-IV")) %>% 
  select(TRIMES, CNUVICAL, codcantf) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>%
  summarise(total = sum(CNUVICAL)) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:15), ~replace(., is.na(.), 0)) %>%
  rename("Quito" = "Distrito Metropolitano De Quito",
         "Samborondón" = "Samborondon",
         "Durán" = "Duran") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CD1_2021)[5] <- "2021"

# 2022
CD1_2022 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2022-I", 
                       "2022-II", 
                       "2022-III", 
                       "2022-IV")) %>% 
  select(TRIMES, CNUVICAL, codcantf) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>%
  summarise(total = sum(CNUVICAL)) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:15), ~replace(., is.na(.), 0)) %>%
  rename("Quito" = "Distrito Metropolitano De Quito",
         "Samborondón" = "Samborondon",
         "Durán" = "Duran") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CD1_2022)[5] <- "2022"

# 2023
CD1_2023 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2023-I", 
                       "2023-II",
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, CNUVICAL, codcantf) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>%
  summarise(total = sum(CNUVICAL)) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:15), ~replace(., is.na(.), 0)) %>%
  rename("Quito" = "Distrito Metropolitano De Quito",
         "Samborondón" = "Samborondon",
         "Durán" = "Duran") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CD1_2023)[5] <- "2023"

# Completo
CD1 <- bind_rows(CD1_2020, CD1_2021, CD1_2022, CD1_2023)

rm(CD1_2019, CD1_2020, CD1_2021, CD1_2022, CD1_2023)


####### C-D2 Viviendas a construir, por trimestres, según metros cuadrados #######

# 2019
CD2_2019 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2019-I", 
                       "2019-II", 
                       "2019-III", 
                       "2019-IV")) %>% 
  select(TRIMES, CNUVICAL, CDISPARC) %>% 
  group_by(TRIMES, as_factor(CDISPARC)) %>%
  summarise(total = sum(CNUVICAL)) %>%
  mutate(subtotal = sum(total)) %>% 
  pivot_wider(names_from = `as_factor(CDISPARC)`, values_from = total) %>% 
  mutate_at(c(2:7), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CD2_2019)[5] <- "2019"

# 2020
CD2_2020 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2020-I", 
                       "2020-II", 
                       "2020-III", 
                       "2020-IV")) %>% 
  select(TRIMES, CNUVICAL, CDISPARC) %>% 
  group_by(TRIMES, as_factor(CDISPARC)) %>%
  summarise(total = sum(CNUVICAL)) %>%
  mutate(subtotal = sum(total)) %>% 
  pivot_wider(names_from = `as_factor(CDISPARC)`, values_from = total) %>% 
  mutate_at(c(2:7), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CD2_2020)[5] <- "2020"

# 2021
CD2_2021 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2021-I", 
                       "2021-II", 
                       "2021-III", 
                       "2021-IV")) %>% 
  select(TRIMES, CNUVICAL, CDISPARC) %>% 
  group_by(TRIMES, as_factor(CDISPARC)) %>%
  summarise(total = sum(CNUVICAL)) %>%
  mutate(subtotal = sum(total)) %>% 
  pivot_wider(names_from = `as_factor(CDISPARC)`, values_from = total) %>% 
  mutate_at(c(2:7), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CD2_2021)[5] <- "2021"

# 2022
CD2_2022 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2022-I", 
                       "2022-II", 
                       "2022-III", 
                       "2022-IV")) %>% 
  select(TRIMES, CNUVICAL, CDISPARC) %>% 
  group_by(TRIMES, as_factor(CDISPARC)) %>%
  summarise(total = sum(CNUVICAL)) %>%
  mutate(subtotal = sum(total)) %>% 
  pivot_wider(names_from = `as_factor(CDISPARC)`, values_from = total) %>% 
  mutate_at(c(2:7), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CD2_2022)[5] <- "2022"

# 2023
CD2_2023 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2023-I" , 
                       "2023-II", 
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, CNUVICAL, CDISPARC) %>% 
  group_by(TRIMES, as_factor(CDISPARC)) %>%
  summarise(total = sum(CNUVICAL)) %>%
  mutate(subtotal = sum(total)) %>% 
  pivot_wider(names_from = `as_factor(CDISPARC)`, values_from = total) %>% 
  mutate_at(c(2:7), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CD2_2023)[5] <- "2023"


# Completo
CD2 <- bind_rows(CD2_2020, CD2_2021, CD2_2022, CD2_2023)
names(CD2) <- c("Total de viviendas",
                "Menos de 100 m2",
                "100 a 199 m2",
                "200 a 299 m2",
                "300 a 399 m2",
                "400 a 499 m2",
                "500 y más m2")
rm(CD2_2019, CD2_2020, CD2_2021, CD2_2022, CD2_2023)


####### C-D3 Viviendas a construir, por trimestres, según número de dormitorios #######

# 2019
CD3_2019 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2019-I", 
                       "2019-II", 
                       "2019-III", 
                       "2019-IV")) %>% 
  select(TRIMES, CNUVICAL, NUDOCAL) %>% 
  mutate(NUDOCAL1 = case_when(NUDOCAL == 0 ~ 0,
                              NUDOCAL == 1 ~ 1,
                              NUDOCAL == 2 ~ 2,
                              NUDOCAL == 3 ~ 3,
                              NUDOCAL == 4 ~ 4,
                              NUDOCAL >= 5 ~ 5,
                              T ~ 0)) %>%
  group_by(TRIMES, as_factor(NUDOCAL1)) %>% 
  summarise(total = sum(CNUVICAL)) %>%
  pivot_wider(names_from = `as_factor(NUDOCAL1)`, values_from = total) %>% 
  mutate_at(c(2:7), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  rename("5 y más" = "5")
rownames(CD3_2019)[5] <- "2019"

# 2020
CD3_2020 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2020-I", 
                       "2020-II", 
                       "2020-III", 
                       "2020-IV")) %>% 
  select(TRIMES, CNUVICAL, NUDOCAL) %>% 
  mutate(NUDOCAL1 = case_when(NUDOCAL == 0 ~ 0,
                              NUDOCAL == 1 ~ 1,
                              NUDOCAL == 2 ~ 2,
                              NUDOCAL == 3 ~ 3,
                              NUDOCAL == 4 ~ 4,
                              NUDOCAL >= 5 ~ 5,
                              T ~ 0)) %>%
  group_by(TRIMES, as_factor(NUDOCAL1)) %>% 
  summarise(total = sum(CNUVICAL)) %>%
  pivot_wider(names_from = `as_factor(NUDOCAL1)`, values_from = total) %>% 
  mutate_at(c(2:7), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  rename("5 y más" = "5")
rownames(CD3_2020)[5] <- "2020"

# 2021
CD3_2021 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2021-I", 
                       "2021-II", 
                       "2021-III", 
                       "2021-IV")) %>% 
  select(TRIMES, CNUVICAL, NUDOCAL) %>% 
  mutate(NUDOCAL1 = case_when(NUDOCAL == 0 ~ 0,
                              NUDOCAL == 1 ~ 1,
                              NUDOCAL == 2 ~ 2,
                              NUDOCAL == 3 ~ 3,
                              NUDOCAL == 4 ~ 4,
                              NUDOCAL >= 5 ~ 5,
                              T ~ 0)) %>%
  group_by(TRIMES, as_factor(NUDOCAL1)) %>% 
  summarise(total = sum(CNUVICAL)) %>%
  pivot_wider(names_from = `as_factor(NUDOCAL1)`, values_from = total) %>% 
  mutate_at(c(2:7), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  rename("5 y más" = "5")
rownames(CD3_2021)[5] <- "2021"

# 2022
CD3_2022 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2022-I", 
                       "2022-II", 
                       "2022-III", 
                       "2022-IV")) %>% 
  select(TRIMES, CNUVICAL, NUDOCAL) %>% 
  mutate(NUDOCAL1 = case_when(NUDOCAL == 0  ~ 0,
                              NUDOCAL >= 1 & NUDOCAL <= 1.49 ~ 1,
                              NUDOCAL >= 1.5 & NUDOCAL <= 2.49 ~ 2,
                              NUDOCAL >= 2.5 & NUDOCAL <= 3.49 ~ 3,
                              NUDOCAL >= 3.5 & NUDOCAL <= 4.49 ~ 4,
                              NUDOCAL >= 4.5 ~ 5,
                              T ~ 0)) %>%
  group_by(TRIMES, as_factor(NUDOCAL1)) %>% 
  summarise(total = sum(CNUVICAL)) %>% 
  pivot_wider(names_from = `as_factor(NUDOCAL1)`, values_from = total) %>% 
  mutate_at(c(2:7), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  rename("5 y más" = "5")
rownames(CD3_2022)[5] <- "2022"

# 2023
CD3_2023 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2023-I",
                       "2023-II", 
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, CNUVICAL, NUDOCAL) %>% 
  mutate(NUDOCAL1 = case_when(NUDOCAL == 0  ~ 0,
                              NUDOCAL >= 1 & NUDOCAL <= 1.49 ~ 1,
                              NUDOCAL >= 1.5 & NUDOCAL <= 2.49 ~ 2,
                              NUDOCAL >= 2.5 & NUDOCAL <= 3.49 ~ 3,
                              NUDOCAL >= 3.5 & NUDOCAL <= 4.49 ~ 4,
                              NUDOCAL >= 4.5 ~ 5,
                              T ~ 0)) %>%
  group_by(TRIMES, as_factor(NUDOCAL1)) %>% 
  summarise(total = sum(CNUVICAL)) %>% 
  pivot_wider(names_from = `as_factor(NUDOCAL1)`, values_from = total) %>% 
  mutate_at(c(2:7), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  rename("5 y más" = "5")
rownames(CD3_2023)[5] <- "2023"

# Completo
CD3 <- bind_rows(CD3_2020, CD3_2021, CD3_2022, CD3_2023) %>% 
  select(-"0")

rm(CD3_2019, CD3_2020, CD3_2021, CD3_2022, CD3_2023)

###### C-E Superficie del terreno y áreas a construir ######* 

####### C-E1 Superficie del terreno y área a construir, por trimestres, según uso de la edificación #######

# 2019
CE1_2019 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2019-I", 
                       "2019-II", 
                       "2019-III", 
                       "2019-IV")) %>% 
  select(TRIMES, CSUTE, CARCO, CARES, CARNRES, CARDPAR, CAESV) %>% 
  group_by(TRIMES) %>% 
  summarise(stt = sum(CSUTE),
            atc = sum(CARCO),
            res = sum(CARES),
            nres = sum(CARNRES),
            par = sum(CARDPAR),
            aev = sum(CAESV)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CE1_2019)[5] <- "2019"

# 2020
CE1_2020 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2020-I", 
                       "2020-II", 
                       "2020-III", 
                       "2020-IV")) %>% 
  select(TRIMES, CSUTE, CARCO, CARES, CARNRES, CARDPAR, CAESV) %>% 
  group_by(TRIMES) %>% 
  summarise(stt = sum(CSUTE),
            atc = sum(CARCO),
            res = sum(CARES),
            nres = sum(CARNRES),
            par = sum(CARDPAR),
            aev = sum(CAESV)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CE1_2020)[5] <- "2020"

# 2021
CE1_2021 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2021-I", 
                       "2021-II", 
                       "2021-III", 
                       "2021-IV")) %>% 
  select(TRIMES, CSUTE, CARCO, CARES, CARNRES, CARDPAR, CAESV) %>% 
  group_by(TRIMES) %>% 
  summarise(stt = sum(CSUTE),
            atc = sum(CARCO),
            res = sum(CARES),
            nres = sum(CARNRES),
            par = sum(CARDPAR),
            aev = sum(CAESV)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CE1_2021)[5] <- "2021"

# 2022
CE1_2022 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2022-I", 
                       "2022-II", 
                       "2022-III", 
                       "2022-IV")) %>% 
  select(TRIMES, CSUTE, CARCO, CARES, CARNRES, CARDPAR, CAESV) %>% 
  group_by(TRIMES) %>% 
  summarise(stt = sum(CSUTE),
            atc = sum(CARCO),
            res = sum(CARES),
            nres = sum(CARNRES),
            par = sum(CARDPAR),
            aev = sum(CAESV)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CE1_2022)[5] <- "2022"

# 2023
CE1_2023 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2023-I" ,
                       "2023-II", 
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, CSUTE, CARCO, CARES, CARNRES, CARDPAR, CAESV) %>% 
  group_by(TRIMES) %>% 
  summarise(stt = sum(CSUTE),
            atc = sum(CARCO),
            res = sum(CARES),
            nres = sum(CARNRES),
            par = sum(CARDPAR),
            aev = sum(CAESV)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CE1_2023)[5] <- "2023"


# Completo
CE1 <- bind_rows(CE1_2020, CE1_2021, CE1_2022, CE1_2023) %>% 
  round(., digits = 0)
names(CE1) <- c("Superficie total del terreno",
                "Total",
                "Residencial",
                "No residencial",
                "Parqueadero*",
                "Área para espacios verdes")

rm(CE1_2019, CE1_2020, CE1_2021, CE1_2022, CE1_2023)


###### C-F Valor estimado de la edificación ######*

####### C-F1 Valor estimado del financiamiento de la edificación, por trimestres, según tipo de obra #######

# 2019
CF1_2019 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2019-I", 
                       "2019-II", 
                       "2019-III", 
                       "2019-IV")) %>% 
  select(TRIMES, CVAE, CTIPOBR) %>% 
  group_by(TRIMES, as_factor(CTIPOBR)) %>% 
  summarise(total = sum(CVAE)) %>% 
  mutate("Monto total de financiamiento" = sum(total)) %>% 
  pivot_wider(names_from = `as_factor(CTIPOBR)`, values_from = total) %>% 
  mutate_at(c(2:5), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CF1_2019)[5] <- "2019"

# 2020
CF1_2020 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2020-I", 
                       "2020-II", 
                       "2020-III", 
                       "2020-IV")) %>% 
  select(TRIMES, CVAE, CTIPOBR) %>% 
  group_by(TRIMES, as_factor(CTIPOBR)) %>% 
  summarise(total = sum(CVAE)) %>% 
  mutate("Monto total de financiamiento" = sum(total)) %>% 
  pivot_wider(names_from = `as_factor(CTIPOBR)`, values_from = total) %>% 
  mutate_at(c(2:5), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CF1_2020)[5] <- "2020"

# 2021
CF1_2021 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2021-I", 
                       "2021-II", 
                       "2021-III", 
                       "2021-IV")) %>% 
  select(TRIMES, CVAE, CTIPOBR) %>% 
  group_by(TRIMES, as_factor(CTIPOBR)) %>% 
  summarise(total = sum(CVAE)) %>% 
  mutate("Monto total de financiamiento" = sum(total)) %>% 
  pivot_wider(names_from = `as_factor(CTIPOBR)`, values_from = total) %>% 
  mutate_at(c(2:5), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CF1_2021)[5] <- "2021"

# 2022
CF1_2022 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2022-I", 
                       "2022-II", 
                       "2022-III", 
                       "2022-IV")) %>% 
  select(TRIMES, CVAE, CTIPOBR) %>% 
  group_by(TRIMES, as_factor(CTIPOBR)) %>% 
  summarise(total = sum(CVAE)) %>% 
  mutate("Monto total de financiamiento" = sum(total)) %>% 
  pivot_wider(names_from = `as_factor(CTIPOBR)`, values_from = total) %>% 
  mutate_at(c(2:5), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CF1_2022)[5] <- "2022"

# 2023
CF1_2023 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2023-I" , 
                       "2023-II",
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, CVAE, CTIPOBR) %>% 
  group_by(TRIMES, as_factor(CTIPOBR)) %>% 
  summarise(total = sum(CVAE)) %>% 
  mutate("Monto total de financiamiento" = sum(total)) %>% 
  pivot_wider(names_from = `as_factor(CTIPOBR)`, values_from = total) %>% 
  mutate_at(c(2:5), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CF1_2023)[5] <- "2023"


# Completo
CF1 <- bind_rows(CF1_2020, CF1_2021, CF1_2022, CF1_2023)

rm(CF1_2019, CF1_2020, CF1_2021, CF1_2022, CF1_2023)


####### C-F2 Valor estimado del financiamiento de la edificación, por trimestres, según uso de la edificación #######

# 2019
CF2_2019 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2019-I", 
                       "2019-II", 
                       "2019-III", 
                       "2019-IV")) %>% 
  select(TRIMES, CVAE, CDISPUSX) %>% 
  group_by(TRIMES, CDISPUSX) %>% 
  summarise(total = sum(CVAE)) %>% 
  complete(CDISPUSX,  fill = list(total = 0)) %>%
  mutate("Monto total de financiamiento" = sum(total)) %>% 
  pivot_wider(names_from = CDISPUSX, values_from = total) %>% 
  mutate_at(c(2:18), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CF2_2019)[5] <- "2019"

# 2020
CF2_2020 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2020-I", 
                       "2020-II", 
                       "2020-III", 
                       "2020-IV")) %>% 
  select(TRIMES, CVAE, CDISPUSX) %>% 
  group_by(TRIMES, CDISPUSX) %>% 
  summarise(total = sum(CVAE)) %>% 
  complete(CDISPUSX,  fill = list(total = 0)) %>%
  mutate("Monto total de financiamiento" = sum(total)) %>% 
  pivot_wider(names_from = CDISPUSX, values_from = total) %>% 
  mutate_at(c(2:18), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CF2_2020)[5] <- "2020"

# 2021
CF2_2021 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2021-I", 
                       "2021-II", 
                       "2021-III", 
                       "2021-IV")) %>% 
  select(TRIMES, CVAE, CDISPUSX) %>% 
  group_by(TRIMES, CDISPUSX) %>% 
  summarise(total = sum(CVAE)) %>% 
  complete(CDISPUSX,  fill = list(total = 0)) %>%
  mutate("Monto total de financiamiento" = sum(total)) %>% 
  pivot_wider(names_from = CDISPUSX, values_from = total) %>% 
  mutate_at(c(2:18), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CF2_2021)[5] <- "2021"

# 2022
CF2_2022 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2022-I", 
                       "2022-II", 
                       "2022-III", 
                       "2022-IV")) %>% 
  select(TRIMES, CVAE, CDISPUSX) %>% 
  group_by(TRIMES, CDISPUSX) %>% 
  summarise(total = sum(CVAE)) %>% 
  complete(CDISPUSX,  fill = list(total = 0)) %>%
  mutate("Monto total de financiamiento" = sum(total)) %>% 
  pivot_wider(names_from = CDISPUSX, values_from = total) %>% 
  mutate_at(c(2:18), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CF2_2022)[5] <- "2022"

# 2023
CF2_2023 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2023-I",
                       "2023-II",
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, CVAE, CDISPUSX) %>% 
  group_by(TRIMES, CDISPUSX) %>% 
  summarise(total = sum(CVAE)) %>% 
  complete(CDISPUSX,  fill = list(total = 0)) %>%
  mutate("Monto total de financiamiento" = sum(total)) %>% 
  pivot_wider(names_from = CDISPUSX, values_from = total) %>% 
  mutate_at(c(2:18), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CF2_2023)[5] <- "2023"

# Completo
CF2 <- bind_rows(CF2_2020, CF2_2021, CF2_2022, CF2_2023) %>% 
  select(- `No Aplica`) %>%
  mutate_at(c(1:16), ~replace(.,is.na(.),0))
names(CF2) <- c("Monto total de financiamiento",
                "Edificaciones con una vivienda",
                "Edificaciones con dos viviendas",
                "Edificaciones con tres o más viviendas",
                "Comercial",
                "Industrial",
                "Edificio administrativo (público)",
                "Educación particular",
                "Educación pública",
                "Cultura",
                "Complejos recreacionales",
                "Hospitales, clínicas y otros establecimientos de salud particular",
                "Hospitales, clínicas y otros establecimientos de salud pública",
                "Transporte y comunicaciones",
                "Mixto residencial y no residencial",
                "Otras")

rm(CF2_2019, CF2_2020, CF2_2021, CF2_2022, CF2_2023)


####### C-F3 Valor estimado de la edificación, por trimestres, según cantones #######

# 2019
CF3_2019 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2019-I", 
                       "2019-II", 
                       "2019-III", 
                       "2019-IV")) %>% 
  select(TRIMES, CVAE, codcantf) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(CVAE)) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>%
  rename("Quito" = "Distrito Metropolitano De Quito",
         "Samborondón" = "Samborondon",
         "Durán" = "Duran") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CF3_2019)[5] <- "2019"

# 2020
CF3_2020 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2020-I", 
                       "2020-II", 
                       "2020-III", 
                       "2020-IV")) %>% 
  select(TRIMES, CVAE, codcantf) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(CVAE)) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>%
  rename("Quito" = "Distrito Metropolitano De Quito",
         "Samborondón" = "Samborondon",
         "Durán" = "Duran") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CF3_2020)[5] <- "2020"

# 2021
CF3_2021 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2021-I", 
                       "2021-II", 
                       "2021-III", 
                       "2021-IV")) %>% 
  select(TRIMES, CVAE, codcantf) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(CVAE)) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>%
  rename("Quito" = "Distrito Metropolitano De Quito",
         "Samborondón" = "Samborondon",
         "Durán" = "Duran") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CF3_2021)[5] <- "2021"

# 2022
CF3_2022 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2022-I", 
                       "2022-II", 
                       "2022-III", 
                       "2022-IV")) %>% 
  select(TRIMES, CVAE, codcantf) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(CVAE)) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>%
  rename("Quito" = "Distrito Metropolitano De Quito",
         "Samborondón" = "Samborondon",
         "Durán" = "Duran") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CF3_2022)[5] <- "2022"

# 2023
CF3_2023 <- ESED_2023_IV %>% 
  filter(TRIMES %in% c("2023-I",
                       "2023-II",
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, CVAE, codcantf) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(CVAE)) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>%
  rename("Quito" = "Distrito Metropolitano De Quito",
         "Samborondón" = "Samborondon",
         "Durán" = "Duran") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CF3_2023)[5] <- "2023"

# Completo
CF3 <- bind_rows(CF3_2020, CF3_2021, CF3_2022, CF3_2023)

rm(CF3_2019, CF3_2020, CF3_2021, CF3_2022, CF3_2023)


# ============================================================================ #
#####               DISEÑO Y EXPORTACIÓN DE TABULADOS A EXCEL              ##### 
# ============================================================================ #

######                Definición de hojas de excel                         #####*

tab_ESED <- createWorkbook()
addWorksheet(tab_ESED, "ÍNDICE", zoom = 60)
addWorksheet(tab_ESED, "C-A1", zoom = 60)
addWorksheet(tab_ESED, "C-B1", zoom = 60)
addWorksheet(tab_ESED, "C-C1", zoom = 60)
addWorksheet(tab_ESED, "C-C2", zoom = 60)
addWorksheet(tab_ESED, "C-C3", zoom = 60)
addWorksheet(tab_ESED, "C-C4", zoom = 60)
addWorksheet(tab_ESED, "C-D1", zoom = 60)
addWorksheet(tab_ESED, "C-D2", zoom = 60)
addWorksheet(tab_ESED, "C-D3", zoom = 60)
addWorksheet(tab_ESED, "C-E1", zoom = 60)
addWorksheet(tab_ESED, "C-F1", zoom = 60)
addWorksheet(tab_ESED, "C-F2", zoom = 60)
addWorksheet(tab_ESED, "C-F3", zoom = 60)

# ============================================================================ #
#########                     Configuración General del Índice          ########
# ============================================================================ #

# Creación del índice

c1 <- c("A", "A1",
        "B", "B1",
        "C", "C1", "C2", "C3", "C4",
        "D", "D1", "D2", "D3",
        "E", "E1",
        "F", "F1", "F2", "F3")

c2 <- c("Cuadro resumen", # A
        "Resultados de las principales variables",
        "Número de permisos de construcción", # B
        "Permisos de construcción, por trimestres, según cantones",
        "Número de edificaciones a construir", # C
        "Edificaciones a construir, por trimestres, según cantones",
        "Edificaciones a construir, por trimestres, según tipo de obra",
        "Edificaciones a construir, por trimestres, según uso de la edificación",
        "Edificaciones a construir, por trimestres, según acceso a internet/celular y servicios básicos",
        "Número de viviendas proyectadas", # D
        "Viviendas a construir, por trimestres, según cantones",
        "Viviendas a construir, por trimestres, según metros cuadrados",
        "Viviendas a construir, por trimestres, según número de dormitorios",
        "Superficie del terreno y áreas a construir", # E
        "Superficie del terreno y área a construir, por trimestres, según uso de la edificación",
        "Valor estimado de la edificación", # F
        "Valor estimado del financiamiento de la edificación, por trimestres, según tipo de obra",
        "Valor estimado del financiamiento de la edificación, por trimestres, según uso de la edificación",
        "Valor estimado de la edificación, por trimestres, según cantones")

ind <- data.frame(c1, c2)		
colnames(ind) <- c("Cuadro No.", "Contenido")

# Diseño y estilos el indice

"Primera fila: Cuadro de Títulos (Cuadro No. - Contenido)"
titenc <- createStyle(fontName = "Century Gothic", 
                      fontSize = 13,
                      textDecoration = "bold",
                      fontColour = "#FFFFFF", 
                      fgFill = "#043F5C",
                      borderColour = "#FFFFFF",
                      halign = "center",
                      valign = "center",
                      border = "TopBottomLeftRight",
                      wrapText = T)

"Filas A-B..."
fill_abcd <- createStyle(fontName = "Century Gothic", 
                         fontSize = 13,
                         textDecoration = "bold",
                         fontColour = "#595959", 
                         fgFill = "#87D3D5",
                         borderColour = "#595959",
                         halign = "left",
                         valign = "center",
                         border = "TopBottomLeftRight",
                         wrapText = T)

"Centrado filas A-B..."
fill_rest1 <- createStyle(fontName = "Century Gothic", 
                         fontSize = 13,
                         textDecoration = "bold",
                         fontColour = "#595959", 
                         fgFill = "#87D3D5",
                         borderColour = "#595959",
                         halign = "center",
                         valign = "center",
                         border = "TopBottomLeftRight",
                         wrapText = T)

"Filas tipo A1-B1..."
fill_a1 <- createStyle(fontName = "Century Gothic",
                       fontSize = 13,
                       fontColour = "#595959", 
                       borderColour = "#595959",
                       halign = "left",
                       valign = "center",
                       border = "TopBottomLeftRight",
                       wrapText = T)

"Centrado filas A1-B1..."
fill_rest2 <- createStyle(fontName = "Century Gothic", 
                          fontSize = 13,
                          fontColour = "#595959", 
                          borderColour = "#595959",
                          halign = "center",
                          valign = "center",
                          border = "TopBottomLeftRight",
                          wrapText = TRUE) 

"Footnotes"
foot1 <- createStyle(fontName = "Century Gothic", 
                    fontSize = 11,
                    fontColour = "#595959",
                    textDecoration = "bold",
                    halign = "left",
                    valign = "center")

foot2 <- createStyle(fontName = "Century Gothic", 
                    fontSize = 11,
                    fontColour = "#595959",
                    halign = "left",
                    valign = "center")

# Configuración 

"Imagen"
insertImage(tab_ESED,
            "ÍNDICE",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\ing_i.png",
            height = 5.82,
            width = 43.08,
            startCol = 2,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "ÍNDICE", 
              rows = 1, 
              heights = "165") 

setRowHeights(tab_ESED, 
              "ÍNDICE", 
              rows = 4:41, 
              heights = "26.25")

"Encabezado"
addStyle(tab_ESED,
         "ÍNDICE",
         titenc,
         cols = 2:3,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "ÍNDICE", 
             cols = 2, 
             widths = "20")

setColWidths(tab_ESED,
             "ÍNDICE", 
             cols = 1, 
             widths = "8")

setColWidths(tab_ESED,
             "ÍNDICE", 
             cols = 3, 
             widths = "208")

"Datos"
writeData(tab_ESED,
          "ÍNDICE",
          ind, 
          startRow = 4, 
          startCol = 2,  
          colNames = T, 
          withFilter = F)

"Diseño filas A-B..."
addStyle(tab_ESED, 
         "ÍNDICE", 
         fill_abcd, 
         cols = 2:3, 
         rows = c(5, 7, 9, 14, 18, 20), 
         gridExpand = T)

"Diseño columna A-B..."
addStyle(tab_ESED,
         "ÍNDICE",
         fill_rest1,
         cols = 2,
         rows = c(5, 7, 9, 14, 18, 20),
         gridExpand = T)

"Diseño filas A1-B1..."
addStyle(tab_ESED,
         "ÍNDICE",
         fill_a1,
         cols = 2:3,
         rows = c(6, 8, 10:13, 15:17, 19, 21:23),
         gridExpand = T)

"Diseño columna A1-B1..."
addStyle(tab_ESED,
         "ÍNDICE",
         fill_rest2,
         cols = 2,
         rows = c(6, 8, 10:13, 15:17, 19, 21:23),
         gridExpand = T)

"Diseño pie de página"
setRowHeights(tab_ESED, 
              "ÍNDICE", 
              rows = 26:27, 
              heights = "12")

setRowHeights(tab_ESED, 
              "ÍNDICE", 
              rows = 26:32, 
              heights = "18")

addStyle(tab_ESED,
         "ÍNDICE",
         foot1,
         cols = 2,
         rows = 26:32,
         gridExpand = T)

addStyle(tab_ESED,
         "ÍNDICE",
         foot2,
         cols = 3,
         rows = 26:32,
         gridExpand = T)
         
writeData(tab_ESED, 
          "ÍNDICE", 
          c("Dirección responsable de la información estadística y contenidos:",
            "Dirección de Estadísticas Económicas"),
          startCol = 2, 
          startRow = 26) 

writeData(tab_ESED, 
          "ÍNDICE", 
          c("Realizadores",
            "Elaboración:",
            "Revisión:",
            "Aprobación:"),
          startCol = 2, 
          startRow = 29) 

writeData(tab_ESED, 
          "ÍNDICE", 
          c("Lorena Ramos / Valery Paz y Miño / Enrique Vallejo",
            "Roberto Chaves",
            "Diana Barco"),
          startCol = 3, 
          startRow = 30) 


# ============================================================================ #
#####################   Configuración  General de los Tabulados   #############
# ============================================================================ #

"Título del cuadro"
tit_c <- createStyle(fontName = "Century Gothic", 
                     fontSize = 14,
                     textDecoration = "bold",
                     fontColour = "#595959",
                     halign = "left",
                     valign = "center",
                     wrapText = T)

"Formato año suma"
for_suma1 <- createStyle(fontName = "Century Gothic", 
                         fontSize = 13,
                         textDecoration = "bold",
                         fontColour = "#595959", 
                         fgFill = "#87D3D5",
                         borderColour = "#595959",
                         halign = "center",
                         valign = "center",
                         border = "TopBottomLeftRight",
                         wrapText = T)

"Formato sumas de datos"
for_suma2 <- createStyle(fontName = "Century Gothic", 
                         fontSize = 13,
                         textDecoration = "bold",
                         fontColour = "#595959", 
                         fgFill = "#87D3D5",
                         borderColour = "#595959",
                         numFmt = "#,##0",
                         halign = "center",
                         valign = "center",
                         border = "TopBottomLeftRight",
                         wrapText = T)

"Formato trimestres datos"
for_dat1 <- createStyle(fontName = "Century Gothic",
                        fontSize = 13,
                        fontColour = "#595959",
                        borderColour = "#595959",
                        halign = "center",
                        valign = "center",
                        border = "TopBottomLeftRight",
                        wrapText = T)

"Formato datos"
for_dat2 <- createStyle(fontName = "Century Gothic",
                        fontSize = 13,
                        fontColour = "#595959",
                        borderColour = "#595959",
                        numFmt = "#,##0",
                        halign = "center",
                        valign = "center",
                        border = "TopBottomLeftRight",
                        wrapText = T)

"Fuente"
fuente <- createStyle(fontName = "Century Gothic", 
                      fontSize = 11,
                      fontColour = "#595959",
                      textDecoration = "bold",
                      halign = "right",
                      valign = "center",
                      wrapText = F)

"Fuente 2 y Nota"
fuente2 <- createStyle(fontName = "Century Gothic",
                       fontSize = 11,
                       fontColour = "#595959",
                       halign = "left",
                       valign = "center",
                       wrapText = F)

# ============================================================================ #
#######################   Configuración Tabulado C-A1   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "C-A1",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\ing_ca1.png",
            height = 5.82,
            width = 36.12,
            startCol = 2,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "C-A1", 
              rows = 1, 
              heights = "165") 

"Título"
writeData(tab_ESED,
          "C-A1",
          "Cuadro A1",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "C-A1",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

setRowHeights(tab_ESED,
              "C-A1",
              rows = c(2, 4),
              heights = "60")

"Encabezado"
addStyle(tab_ESED,
         "C-A1",
         titenc,
         cols = 2:8,
         rows = 4,
         gridExpand = T)

"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "C-A1", 
             cols = 2:8, 
             widths = "27")

setColWidths(tab_ESED,
             "C-A1", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "C-A1",
              rows = 5:24,
              heights = "40") # cambiar

"Datos y Diseño"
writeData(tab_ESED,
          "C-A1",
          CA1,
          startCol = 2,
          startRow = 4,
          colNames = T,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "C-A1",
          "Trimestres",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "C-A1",
         for_dat1,
         cols = 2,
         rows = c(5:8, 10:13, 15:18, 20:23), # cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C-A1",
         for_dat2,
         cols = 3:8,
         rows = c(5:8, 10:13, 15:18, 20:23), # cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C-A1",
         for_suma1,
         cols = 2,
         rows = c(9, 14, 19 , 24),
         gridExpand = T)

addStyle(tab_ESED,
         "C-A1",
         for_suma2,
         cols = 3:8,
         rows = c(9, 14, 19 , 24 ),
         gridExpand = T)

"Fuentes y notas"

setRowHeights(tab_ESED, 
              "C-A1", 
              rows = 26:27, # cambiar
              heights = "18")

writeData(tab_ESED,
          "C-A1",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2020 - 2023",
          startCol = 2,
          startRow = 26) # cambiar

addStyle(tab_ESED,
         "C-A1",
         fuente2,
         cols = 2,
         rows = 26, # cambiar
         gridExpand = T)

# ============================================================================ #
#######################   Configuración Tabulado C-B1   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "C-B1",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\ing_cb1.png",
            height = 5.82,
            width = 65.3,
            startCol = 2,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "C-B1", 
              rows = 1, 
              heights = "165") 

"Título"
writeData(tab_ESED,
          "C-B1",
          "Cuadro B1",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "C-B1",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

setRowHeights(tab_ESED,
              "C-B1",
              rows = c(2, 4),
              heights = "60")

"Encabezado"
addStyle(tab_ESED,
         "C-B1",
         titenc,
         cols = 2:17,
         rows = 4,
         gridExpand = T)


"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "C-B1", 
             cols = 2, 
             widths = "35")

setColWidths(tab_ESED,
             "C-B1", 
             cols = 3:17, 
             widths = "20")

setColWidths(tab_ESED,
             "C-B1", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "C-B1",
              rows = 5:24,  # cambiar
              heights = "40")


"Datos y Diseño"
writeData(tab_ESED,
          "C-B1",
          CB1,
          startCol = 2,
          startRow = 4,
          colNames = T,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "C-B1",
          "Trimestres / Cantones",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "C-B1",
         for_dat1,
         cols = 2,
         rows = c(5:8, 10:13, 15:18, 20:23), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C-B1",
         for_dat2,
         cols = 3:17,
         rows = c(5:8, 10:13, 15:18, 20:23), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C-B1",
         for_suma1,
         cols = 2,
         rows = c(9, 14, 19 , 24 ),
         gridExpand = T)

addStyle(tab_ESED,
         "C-B1",
         for_suma2,
         cols = 3:17,
         rows = c(9, 14, 19 , 24 ),
         gridExpand = T)

"Fuentes y notas"
setRowHeights(tab_ESED, 
              "C-B1", 
              rows = 26:27, # cambiar
              heights = "18")

writeData(tab_ESED,
          "C-B1",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2020 - 2023",
          startCol = 2,
          startRow = 26) # cambiar

addStyle(tab_ESED,
         "C-B1",
         fuente2,
         cols = 2,
         rows = 26, # cambiar
         gridExpand = T)


# ============================================================================ #
#######################   Configuración Tabulado C-C1   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "C-C1",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\ing_cc1.png",
            height = 5.82,
            width = 65.3,
            startCol = 2,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "C-C1", 
              rows = 1, 
              heights = "165") 

"Título"
writeData(tab_ESED,
          "C-C1",
          "Cuadro C1",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "C-C1",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

setRowHeights(tab_ESED,
              "C-C1",
              rows = c(2, 4),
              heights = "60")

"Encabezado"
addStyle(tab_ESED,
         "C-C1",
         titenc,
         cols = 2:17,
         rows = 4,
         gridExpand = T)

"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "C-C1", 
             cols = 2, 
             widths = "35")

setColWidths(tab_ESED,
             "C-C1", 
             cols = 3:17, 
             widths = "20")

setColWidths(tab_ESED,
             "C-C1", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "C-C1",
              rows = 5:24,
              heights = "40") # cambiar

"Datos y Diseño"
writeData(tab_ESED,
          "C-C1",
          CC1,
          startCol = 2,
          startRow = 4,
          colNames = T,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "C-C1",
          "Trimestres / Cantones",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "C-C1",
         for_dat1,
         cols = 2,
         rows = c(5:8, 10:13, 15:18, 20:23), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C-C1",
         for_dat2,
         cols = 3:17,
         rows = c(5:8, 10:13, 15:18, 20:23), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C-C1",
         for_suma1,
         cols = 2,
         rows = c(9, 14, 19 , 24 ),
         gridExpand = T)

addStyle(tab_ESED,
         "C-C1",
         for_suma2,
         cols = 3:17,
         rows = c(9, 14, 19 , 24 ),
         gridExpand = T)

"Fuentes y notas"
setRowHeights(tab_ESED, 
             "C-C1", 
             rows = 26:27, # cambiar
             heights = "18")

writeData(tab_ESED,
          "C-C1",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2020 - 2023",
          startCol = 2,
          startRow = 26) # cambiar

addStyle(tab_ESED,
         "C-C1",
         fuente2,
         cols = 2,
         rows = 26, # cambiar
         gridExpand = T)

# ============================================================================ #
#######################   Configuración Tabulado C-C2   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "C-C2",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\ing_cc2.png",
            height = 5.82,
            width = 33.38,
            startCol = 2,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "C-C2", 
              rows = 1, 
              heights = "165") 

"Título"
writeData(tab_ESED,
          "C-C2",
          "Cuadro C2",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "C-C2",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

setRowHeights(tab_ESED,
              "C-C2",
              rows = 2,
              heights = "60")

"Encabezado"
addStyle(tab_ESED,
         "C-C2",
         titenc,
         cols = 2:6,
         rows = 4:5,
         gridExpand = T)

"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "C-C2", 
             cols = 2, 
             widths = "35")

setColWidths(tab_ESED,
             "C-C2", 
             cols = 3:6, 
             widths = "35")

setColWidths(tab_ESED,
             "C-C2", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "C-C2",
              rows = 6:25,
              heights = "40") #cambiar

setRowHeights(tab_ESED,
              "C-C2",
              rows = 4:5,
              heights = "30")

"Celdas de títulos"
mergeCells(tab_ESED,
           "C-C2",
           cols = 2,
           rows = 4:5)

mergeCells(tab_ESED,
           "C-C2",
           cols = 3,
           rows = 4:5)

mergeCells(tab_ESED,
           "C-C2",
           cols = 4:6,
           rows = 4)

"Datos y Diseño"
writeData(tab_ESED,
          "C-C2",
          CC2,
          startCol = 2,
          startRow = 5,
          colNames = T,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "C-C2",
          "Trimestres",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "C-C2",
         for_dat1,
         cols = 2,
         rows = c(6:9, 11:14, 16:19, 21:24), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C-C2",
         for_dat2,
         cols = 3:6,
         rows = c(6:9, 11:14, 16:19, 21:24), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C-C2",
         for_suma1,
         cols = 2,
         rows = c(10, 15, 20 , 25 ), 
         gridExpand = T)

addStyle(tab_ESED,
         "C-C2",
         for_suma2,
         cols = 3:6,
         rows = c(10, 15, 20 , 25 ), 
         gridExpand = T)

"Arreglo títulos"
writeData(tab_ESED,
          "C-C2",
          "Tipo de obra",
          startCol = 4,
          startRow = 4)

writeData(tab_ESED,
          "C-C2",
          "Total de edificaciones",
          startCol = 3,
          startRow = 4)

"Fuentes y notas"
setRowHeights(tab_ESED, 
              "C-C2", 
              rows = 27:28,  #cambiar
              heights = "18")

writeData(tab_ESED,
          "C-C2",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2020 - 2023",
          startCol = 2,
          startRow = 27) #cambiar

addStyle(tab_ESED,
         "C-C2",
         fuente2,
         cols = 2,
         rows = 27, #cambiar
         gridExpand = T)

# ============================================================================ #
#######################   Configuración Tabulado C-C3   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "C-C3",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\ing_cc3.png",
            height = 5.82,
            width = 93,
            startCol = 2,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "C-C3", 
              rows = 1, 
              heights = "165") 

"Título"
writeData(tab_ESED,
          "C-C3",
          "Cuadro C3",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "C-C3",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

setRowHeights(tab_ESED,
              "C-C3",
              rows = 2,
              heights = "60")

"Encabezado"
addStyle(tab_ESED,
         "C-C3",
         titenc,
         cols = 2:17,
         rows = 4,
         gridExpand = T)

"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "C-C3", 
             cols = 2, 
             widths = "35")

setColWidths(tab_ESED,
             "C-C3", 
             cols = 3:17, 
             widths = "30")

setColWidths(tab_ESED,
             "C-C3", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "C-C3",
              rows = 5:24,
              heights = "40") #cambiar

setRowHeights(tab_ESED,
              "C-C3",
              rows = 4,
              heights = "70")

"Datos y Diseño"
writeData(tab_ESED,
          "C-C3",
          CC3,
          startCol = 2,
          startRow = 4,
          colNames = T,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "C-C3",
          "Trimestres",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "C-C3",
         for_dat1,
         cols = 2,
         rows = c(5:8, 10:13, 15:18, 20:23), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C-C3",
         for_dat2,
         cols = 3:17,
         rows = c(5:8, 10:13, 15:18, 20:23), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C-C3",
         for_suma1,
         cols = 2,
         rows = c(9, 14, 19 , 24 ),
         gridExpand = T)

addStyle(tab_ESED,
         "C-C3",
         for_suma2,
         cols = 3:17,
         rows = c(9, 14, 19 , 24 ),
         gridExpand = T)

"Fuentes y notas"
setRowHeights(tab_ESED, 
              "C-C3",
              rows = 26:27, # cambiar
              heights = "18")

writeData(tab_ESED,
          "C-C3",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2020 - 2023",
          startCol = 2,
          startRow = 26) # cambiar

addStyle(tab_ESED,
         "C-C3",
         fuente2,
         cols = 2,
         rows = 26, # cambiar
         gridExpand = T)

# ============================================================================ #
#######################   Configuración Tabulado C-C4   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "C-C4",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\ing_cc4.png",
            height = 5.5,
            width = 27.91 ,
            startCol = 2,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "C-C4", 
              rows = 1, 
              heights = "165") 

"Título"
writeData(tab_ESED,
          "C-C4",
          "Cuadro C4",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "C-C4",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

setRowHeights(tab_ESED,
              "C-C4",
              rows = 2,
              heights = "60")

"Encabezado"
addStyle(tab_ESED,
         "C-C4",
         titenc,
         cols = 2:6,
         rows = 4:6,
         gridExpand = T)

"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "C-C4", 
             cols = 2, 
             widths = "25")

setColWidths(tab_ESED,
             "C-C4", 
             cols = 3:6, 
             widths = "30")

setColWidths(tab_ESED,
             "C-C4", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "C-C4",
              rows = 7:26,
              heights = "40")#cambiar

setRowHeights(tab_ESED,
              "C-C4",
              rows = 4:6,
              heights = "30")

"Celdas de títulos"
mergeCells(tab_ESED,
           "C-C4",
           cols = 2,
           rows = 4:6)

mergeCells(tab_ESED,
           "C-C4",
           cols = 3:6,
           rows = 4)

mergeCells(tab_ESED,
           "C-C4",
           cols = 3:4,
           rows = 5)

mergeCells(tab_ESED,
           "C-C4",
           cols = 5:6,
           rows = 5)

"Datos y Diseño"
writeData(tab_ESED,
          "C-C4",
          CC4,
          startCol = 2,
          startRow = 7,
          colNames = F,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "C-C4",
          "Trimestres",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "C-C4",
         for_dat1,
         cols = 2,
         rows = c(7:10, 12:15, 17:20, 22:25), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C-C4",
         for_dat2,
         cols = 3:6,
         rows = c(7:10, 12:15, 17:20, 22:25), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C-C4",
         for_suma1,
         cols = 2,
         rows = c(11, 16, 21 , 26),
         gridExpand = T)

addStyle(tab_ESED,
         "C-C4",
         for_suma2,
         cols = 3:6,
         rows = c(11, 16, 21 , 26),
         gridExpand = T)

"Arreglo títulos"
writeData(tab_ESED,
          "C-C4",
          "Internet / celular y servicios básicos",
          startCol = 3,
          startRow = 4)

writeData(tab_ESED,
          "C-C4",
          "Área urbana",
          startCol = 3,
          startRow = 5)

writeData(tab_ESED,
          "C-C4",
          "Área rural",
          startCol = 5,
          startRow = 5)

writeData(tab_ESED,
          "C-C4",
          "Con acceso",
          startCol = 3,
          startRow = 6)

writeData(tab_ESED,
          "C-C4",
          "Sin acceso",
          startCol = 4,
          startRow = 6)

writeData(tab_ESED,
          "C-C4",
          "Con acceso",
          startCol = 5,
          startRow = 6)

writeData(tab_ESED,
          "C-C4",
          "Sin acceso",
          startCol = 6,
          startRow = 6)


"Fuentes y notas"

setRowHeights(tab_ESED, 
              "C-C4" , 
              rows = 28:29,  #cambiar
              heights = "18")

writeData(tab_ESED,
          "C-C4",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2020 - 2023",
          startCol = 2,
          startRow = 28) #cambiar

addStyle(tab_ESED,
         "C-C4",
         fuente2,
         cols = 2,
         rows = 28, #cambiar
         gridExpand = T)

writeData(tab_ESED,
          "C-C4",
          "Nota: Se excluye la categoría (No responde) para la construcción de este tabulado",
          startCol = 2,
          startRow = 29) #cambiar

addStyle(tab_ESED,
         "C-C4",
         fuente2,
         cols = 2,
         rows = 29) #cambiar

# ============================================================================ #
#######################   Configuración Tabulado C-D1   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "C-D1",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\ing_cd1.png",
            height = 5.82,
            width = 65.3,
            startCol = 2,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "C-D1", 
              rows = 1, 
              heights = "165") 

"Título"
writeData(tab_ESED,
          "C-D1",
          "Cuadro D1",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "C-D1",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

setRowHeights(tab_ESED,
              "C-D1",
              rows = c(2, 4),
              heights = "60")

"Encabezado"
addStyle(tab_ESED,
         "C-D1",
         titenc,
         cols = 2:17,
         rows = 4,
         gridExpand = T)

"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "C-D1", 
             cols = 2, 
             widths = "35")

setColWidths(tab_ESED,
             "C-D1", 
             cols = 3:17, 
             widths = "20")

setColWidths(tab_ESED,
             "C-D1", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "C-D1",
              rows = 5:24,
              heights = "40")#cambiar

"Datos y Diseño"
writeData(tab_ESED,
          "C-D1",
          CD1,
          startCol = 2,
          startRow = 4,
          colNames = T,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "C-D1",
          "Trimestres / Cantones",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "C-D1",
         for_dat1,
         cols = 2,
         rows = c(5:8, 10:13, 15:18, 20:23), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C-D1",
         for_dat2,
         cols = 3:17,
         rows = c(5:8, 10:13, 15:18, 20:23), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C-D1",
         for_suma1,
         cols = 2,
         rows = c(9, 14, 19 , 24 ),
         gridExpand = T)

addStyle(tab_ESED,
         "C-D1",
         for_suma2,
         cols = 3:17,
         rows = c(9, 14, 19 , 24 ),
         gridExpand = T)

"Fuentes y notas"
setRowHeights(tab_ESED, 
              "C-D1",
              rows = 26:27, # cambiar
              heights = "18")

writeData(tab_ESED,
          "C-D1",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2020 - 2023",
          startCol = 2,
          startRow = 26) # cambiar

addStyle(tab_ESED,
         "C-D1",
         fuente2,
         cols = 2,
         rows = 26, # cambiar
         gridExpand = T)

# ============================================================================ #
#######################   Configuración Tabulado C-D2   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "C-D2",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\ing_cd2.png",
            height = 5.82,
            width = 46.95,
            startCol = 2,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "C-D2", 
              rows = 1, 
              heights = "165") 

"Título"
writeData(tab_ESED,
          "C-D2",
          "Cuadro D2",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "C-D2",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

setRowHeights(tab_ESED,
              "C-D2",
              rows = 2,
              heights = "60")

"Encabezado"
addStyle(tab_ESED,
         "C-D2",
         titenc,
         cols = 2:9,
         rows = 4:5,
         gridExpand = T)

"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "C-D2", 
             cols = 2, 
             widths = "35")

setColWidths(tab_ESED,
             "C-D2", 
             cols = 3:9, 
             widths = "30")

setColWidths(tab_ESED,
             "C-D2", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "C-D2",
              rows = 6:25,
              heights = "40") #cambiar

setRowHeights(tab_ESED,
              "C-D2",
              rows = 4:5,
              heights = "30")

"Celdas de títulos"
mergeCells(tab_ESED,
           "C-D2",
           cols = 2,
           rows = 4:5)

mergeCells(tab_ESED,
           "C-D2",
           cols = 3,
           rows = 4:5)

mergeCells(tab_ESED,
           "C-D2",
           cols = 4:9,
           rows = 4)

"Datos y Diseño"
writeData(tab_ESED,
          "C-D2",
          CD2,
          startCol = 2,
          startRow = 5,
          colNames = T,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "C-D2",
          "Trimestres",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "C-D2",
         for_dat1,
         cols = 2,
         rows = c(6:9, 11:14, 16:19, 21:24), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C-D2",
         for_dat2,
         cols = 3:9,
         rows = c(6:9, 11:14, 16:19, 21:24), #cambiar
         gridExpand = T) 

addStyle(tab_ESED,
         "C-D2",
         for_suma1,
         cols = 2,
         rows = c(10, 15, 20 , 25 ), 
         gridExpand = T)

addStyle(tab_ESED,
         "C-D2",
         for_suma2,
         cols = 3:9,
         rows = c(10, 15, 20 , 25 ), 
         gridExpand = T)

"Arreglo títulos"
writeData(tab_ESED,
          "C-D2",
          "Rangos de área total a construir",
          startCol = 4,
          startRow = 4)

writeData(tab_ESED,
          "C-D2",
          "Total de viviendas",
          startCol = 3,
          startRow = 4)

"Fuentes y notas"

setRowHeights(tab_ESED, 
              "C-D2",
              rows = 27:28,  #cambiar
              heights = "18")

writeData(tab_ESED,
          "C-D2",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2020 - 2023",
          startCol = 2,
          startRow = 27) #cambiar

addStyle(tab_ESED,
         "C-D2",
         fuente2,
         cols = 2,
         rows = 27, #cambiar
         gridExpand = T)

# ============================================================================ #
#######################   Configuración Tabulado C-D3   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "C-D3",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\ing_cd3.png",
            height = 5.82,
            width = 30.78,
            startCol = 2,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "C-D3", 
              rows = 1, 
              heights = "165") 

"Título"
writeData(tab_ESED,
          "C-D3",
          "Cuadro D3",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "C-D3",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

setRowHeights(tab_ESED,
              "C-D3",
              rows = 2,
              heights = "60")

"Encabezado"
addStyle(tab_ESED,
         "C-D3",
         titenc,
         cols = 2:7,
         rows = 4:5,
         gridExpand = T)

"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "C-D3", 
             cols = 2, 
             widths = "35")

setColWidths(tab_ESED,
             "C-D3", 
             cols = 3:7, 
             widths = "25")

setColWidths(tab_ESED,
             "C-D3", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "C-D3",
              rows = 6:25 ,
              heights = "40") # cambiar

setRowHeights(tab_ESED,
              "C-D3",
              rows = 4:5,
              heights = "30")

"Celdas de títulos"
mergeCells(tab_ESED,
           "C-D3",
           cols = 2,
           rows = 4:5)

mergeCells(tab_ESED,
           "C-D3",
           cols = 3:7,
           rows = 4)

"Datos y Diseño"
writeData(tab_ESED,
          "C-D3",
          CD3,
          startCol = 2,
          startRow = 5,
          colNames = T,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "C-D3",
          "Trimestres",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "C-D3",
         for_dat1,
         cols = 2,
         rows = c(6:9, 11:14, 16:19, 21:24), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C-D3",
         for_dat2,
         cols = 3:7,
         rows = c(6:9, 11:14, 16:19, 21:24), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C-D3",
         for_suma1,
         cols = 2,
         rows = c(10, 15, 20 , 25 ), 
         gridExpand = T)

addStyle(tab_ESED,
         "C-D3",
         for_suma2,
         cols = 3:7,
         rows = c(10, 15, 20 , 25 ), 
         gridExpand = T)

"Arreglo títulos"
writeData(tab_ESED,
          "C-D3",
          "Número de dormitorios por vivienda",
          startCol = 3,
          startRow = 4)

"Fuentes y notas"

setRowHeights(tab_ESED, 
              "C-D3",
              rows = 27:28, #cambiar
              heights = "18")

writeData(tab_ESED,
          "C-D3",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2020 - 2023",
          startCol = 2,
          startRow = 27) #cambiar

addStyle(tab_ESED,
         "C-D3",
         fuente2,
         cols = 2,
         rows = 27, #cambiar
         gridExpand = T)


writeData(tab_ESED,
          "C-D3",
          "Nota: Para la construcción de este tabulado se excluye las reconstrucciones",
          startCol = 2,
          startRow = 28) #cambiar

addStyle(tab_ESED,
         "C-D3",
         fuente2,
         cols = 2,
         rows = 28) #cambiar

# ============================================================================ #
#######################   Configuración Tabulado C-E1   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "C-E1",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\ing_ce1.png",
            height = 5.82,
            width = 41.14,
            startCol = 2,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "C-E1", 
              rows = 1, 
              heights = "165") 

"Título"
writeData(tab_ESED,
          "C-E1",
          "Cuadro E1",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "C-E1",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

setRowHeights(tab_ESED,
              "C-E1",
              rows = 2,
              heights = "60")

"Encabezado"
addStyle(tab_ESED,
         "C-E1",
         titenc,
         cols = 2:8,
         rows = 4:5,
         gridExpand = T)

"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "C-E1", 
             cols = 2, 
             widths = "35")

setColWidths(tab_ESED,
             "C-E1", 
             cols = 3:8, 
             widths = "30")

setColWidths(tab_ESED,
             "C-E1", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "C-E1",
              rows = 6:25,
              heights = "40")#cambiar

setRowHeights(tab_ESED,
              "C-E1",
              rows = 4:5,
              heights = "30")

"Celdas de títulos"
mergeCells(tab_ESED,
           "C-E1",
           cols = 2,
           rows = 4:5)

mergeCells(tab_ESED,
           "C-E1",
           cols = 3,
           rows = 4:5)

mergeCells(tab_ESED,
           "C-E1",
           cols = 4:7,
           rows = 4)

mergeCells(tab_ESED,
           "C-E1",
           cols = 8,
           rows = 4:5)

"Datos y Diseño"
writeData(tab_ESED,
          "C-E1",
          CE1,
          startCol = 2,
          startRow = 5,
          colNames = T,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "C-E1",
          "Trimestres",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "C-E1",
         for_dat1,
         cols = 2,
         rows = c(6:9, 11:14, 16:19, 21:24),#cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C-E1",
         for_dat2,
         cols = 3:8,
         rows = c(6:9, 11:14, 16:19, 21:24), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C-E1",
         for_suma1,
         cols = 2,
         rows = c(10, 15, 20 , 25 ), 
         gridExpand = T)

addStyle(tab_ESED,
         "C-E1",
         for_suma2,
         cols = 3:8,
         rows = c(10, 15, 20 , 25 ), 
         gridExpand = T)

"Arreglo títulos"
writeData(tab_ESED,
          "C-E1",
          "Área a construir",
          startCol = 4,
          startRow = 4)

writeData(tab_ESED,
          "C-E1",
          "Superficie total del terreno",
          startCol = 3,
          startRow = 4)

writeData(tab_ESED,
          "C-E1",
          "Área para espacios verdes",
          startCol = 8,
          startRow = 4)

"Fuentes y notas"

setRowHeights(tab_ESED, 
              "C-E1", 
              rows = 27:28, # cambiar
              heights = "18")

writeData(tab_ESED,
          "C-E1",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2020 - 2023",
          startCol = 2,
          startRow = 27) # cambiar

addStyle(tab_ESED,
         "C-E1",
         fuente2,
         cols = 2,
         rows = 27, # cambiar
         gridExpand = T)


writeData(tab_ESED,
          "C-E1",
          "*Parqueaderos de las edificaciones con fines no residenciales",
          startCol = 2,
          startRow = 28) # cambiar

addStyle(tab_ESED,
         "C-E1",
         fuente2,
         cols = 2,
         rows = 28) # cambiar


# ============================================================================ #
#######################   Configuración Tabulado C-F1   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "C-F1",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\ing_cf1.png",
            height = 5.82,
            width = 33.34,
            startCol = 2,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "C-F1", 
              rows = 1, 
              heights = "165") 

"Título"
writeData(tab_ESED,
          "C-F1",
          "Cuadro F1",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "C-F1",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

setRowHeights(tab_ESED,
              "C-F1",
              rows = 2,
              heights = "60")

"Encabezado"
addStyle(tab_ESED,
         "C-F1",
         titenc,
         cols = 2:6,
         rows = 4:5,
         gridExpand = T)

"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "C-F1", 
             cols = 2, 
             widths = "35")

setColWidths(tab_ESED,
             "C-F1", 
             cols = 3:6, 
             widths = "35")

setColWidths(tab_ESED,
             "C-F1", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "C-F1",
              rows = 6:25,
              heights = "40")#cambiar

setRowHeights(tab_ESED,
              "C-F1",
              rows = 4:5,
              heights = "30")

"Celdas de títulos"
mergeCells(tab_ESED,
           "C-F1",
           cols = 2,
           rows = 4:5)

mergeCells(tab_ESED,
           "C-F1",
           cols = 3,
           rows = 4:5)

mergeCells(tab_ESED,
           "C-F1",
           cols = 4:6,
           rows = 4)

"Datos y Diseño"
writeData(tab_ESED,
          "C-F1",
          CF1,
          startCol = 2,
          startRow = 5,
          colNames = T,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "C-F1",
          "Trimestres",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "C-F1",
         for_dat1,
         cols = 2,
         rows = c(6:9, 11:14, 16:19, 21:24), # cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C-F1",
         for_dat2,
         cols = 3:6,
         rows = c(6:9, 11:14, 16:19, 21:24), # cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C-F1",
         for_suma1,
         cols = 2,
         rows = c(10, 15, 20 , 25 ), 
         gridExpand = T)

addStyle(tab_ESED,
         "C-F1",
         for_suma2,
         cols = 3:6,
         rows = c(10, 15, 20 , 25 ), 
         gridExpand = T)

"Arreglo títulos"
writeData(tab_ESED,
          "C-F1",
          "Tipo de obra",
          startCol = 4,
          startRow = 4)

writeData(tab_ESED,
          "C-F1",
          "Monto total de financiamiento",
          startCol = 3,
          startRow = 4)

"Fuentes y notas"

setRowHeights(tab_ESED, 
              "C-F1",
              rows = 27:28,  # cambiar
              heights = "18")

writeData(tab_ESED,
          "C-F1",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2020 - 2023 ",
          startCol = 2,
          startRow = 27) # cambiar

addStyle(tab_ESED,
         "C-F1",
         fuente2,
         cols = 2,
         rows = 27, # cambiar
         gridExpand = T)

# ============================================================================ #
#######################   Configuración Tabulado C-F2   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "C-F2",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\ing_cf2.png",
            height = 5.82,
            width = 98.82,
            startCol = 2,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "C-F2", 
              rows = 1, 
              heights = "165") 

"Título"
writeData(tab_ESED,
          "C-F2",
          "Cuadro F2",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "C-F2",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

setRowHeights(tab_ESED,
              "C-F2",
              rows = 2,
              heights = "60")

"Encabezado"
addStyle(tab_ESED,
         "C-F2",
         titenc,
         cols = 2:18,
         rows = 4,
         gridExpand = T)

"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "C-F2", 
             cols = 2, 
             widths = "35")

setColWidths(tab_ESED,
             "C-F2", 
             cols = 3:18, 
             widths = "30")

setColWidths(tab_ESED,
             "C-F2", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "C-F2",
              rows = 5:24,
              heights = "40")#cambiar

setRowHeights(tab_ESED,
              "C-F2",
              rows = 4,
              heights = "70")

"Datos y Diseño"
writeData(tab_ESED,
          "C-F2",
          CF2,
          startCol = 2,
          startRow = 4,
          colNames = T,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "C-F2",
          "Trimestres",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "C-F2",
         for_dat1,
         cols = 2,
         rows = c(5:8, 10:13, 15:18, 20:23),
         gridExpand = T)

addStyle(tab_ESED,
         "C-F2",
         for_dat2,
         cols = 3:18,
         rows = c(5:8, 10:13, 15:18, 20:23),
         gridExpand = T)

addStyle(tab_ESED,
         "C-F2",
         for_suma1,
         cols = 2,
         rows = c(9, 14, 19 , 24 ),
         gridExpand = T)

addStyle(tab_ESED,
         "C-F2",
         for_suma2,
         cols = 3:18,
         rows = c(9, 14, 19 , 24 ),
         gridExpand = T)

"Fuentes y notas"
setRowHeights(tab_ESED, 
              "C-F2",
              rows = 26:27, # cambiar
              heights = "18")

writeData(tab_ESED,
          "C-F2",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2020 - 2023",
          startCol = 2,
          startRow = 26) # cambiar

addStyle(tab_ESED,
         "C-F2",
         fuente2,
         cols = 2,
         rows = 26, # cambiar
         gridExpand = T)

# ============================================================================ #
#######################   Configuración Tabulado C-F3   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "C-F3",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\ing_cf3.png",
            height = 5.82,
            width = 65.3,
            startCol = 2,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "C-F3", 
              rows = 1, 
              heights = "165") 

"Título"
writeData(tab_ESED,
          "C-F3",
          "Cuadro F3",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "C-F3",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

setRowHeights(tab_ESED,
              "C-F3",
              rows = c(2, 4),
              heights = "60")

"Encabezado"
addStyle(tab_ESED,
         "C-F3",
         titenc,
         cols = 2:17,
         rows = 4,
         gridExpand = T)

"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "C-F3", 
             cols = 2, 
             widths = "35")

setColWidths(tab_ESED,
             "C-F3", 
             cols = 3:17, 
             widths = "20")

setColWidths(tab_ESED,
             "C-F3", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "C-F3",
              rows = 5:24,
              heights = "40")#cambiar

"Datos y Diseño"
writeData(tab_ESED,
          "C-F3",
          CF3,
          startCol = 2,
          startRow = 4,
          colNames = T,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "C-F3",
          "Trimestres / Cantones",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "C-F3",
         for_dat1,
         cols = 2,
         rows = c(5:8, 10:13, 15:18, 20:23),
         gridExpand = T)

addStyle(tab_ESED,
         "C-F3",
         for_dat2,
         cols = 3:17,
         rows = c(5:8, 10:13, 15:18, 20:23),
         gridExpand = T)

addStyle(tab_ESED,
         "C-F3",
         for_suma1,
         cols = 2,
         rows = c(9, 14, 19 , 24 ),
         gridExpand = T)

addStyle(tab_ESED,
         "C-F3",
         for_suma2,
         cols = 3:17,
         rows = c(9, 14, 19 , 24 ),
         gridExpand = T)

"Fuentes y notas"
setRowHeights(tab_ESED, 
              "C-F3",
              rows = 26:27, # cambiar
              heights = "18")

writeData(tab_ESED,
          "C-F3",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2020 - 2023",
          startCol = 2,
          startRow = 26) # cambiar

addStyle(tab_ESED,
         "C-F3",
         fuente2,
         cols = 2,
         rows = 26, # cambiar
         gridExpand = T)

# ============================================================================ #
##################   Creación de hipervínculos en el Índice   ##################
# ============================================================================ #

"C-A1"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 6,
             x = makeHyperlinkString(
               sheet = "C-A1",
               row = 1,
               col = 1,
               text = "Resultados de las principales variables"))

"C-B1"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 8,
             x = makeHyperlinkString(
               sheet = "C-B1",
               row = 1,
               col = 1,
               text = "Permisos de construcción, por trimestres, según cantones"))

"C-C1"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 10,
             x = makeHyperlinkString(
               sheet = "C-C1",
               row = 1,
               col = 1,
               text = "Edificaciones a construir, por trimestres, según cantones"))

"C-C2"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 11,
             x = makeHyperlinkString(
               sheet = "C-C2",
               row = 1,
               col = 1,
               text = "Edificaciones a construir, por trimestres, según tipo de obra"))

"C-C3"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 12,
             x = makeHyperlinkString(
               sheet = "C-C3",
               row = 1,
               col = 1,
               text = "Edificaciones a construir, por trimestres, según uso de la edificación"))

"C-C4"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 13,
             x = makeHyperlinkString(
               sheet = "C-C4",
               row = 1,
               col = 1,
               text = "Edificaciones a construir, por trimestres, según acceso a internet/celular y servicios básicos"))

"C-D1"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 15,
             x = makeHyperlinkString(
               sheet = "C-D1",
               row = 1,
               col = 1,
               text = "Viviendas a construir, por trimestres, según cantones"))

"C-D2"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 16,
             x = makeHyperlinkString(
               sheet = "C-D2",
               row = 1,
               col = 1,
               text = "Viviendas a construir, por trimestres, según metros cuadrados"))

"C-D3"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 17,
             x = makeHyperlinkString(
               sheet = "C-D3",
               row = 1,
               col = 1,
               text = "Viviendas a construir, por trimestres, según número de dormitorios"))

"C-E1"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 19,
             x = makeHyperlinkString(
               sheet = "C-E1",
               row = 1,
               col = 1,
               text = "Superficie del terreno y área a construir, por trimestres, según uso de la edificación"))

"C-F1"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 21,
             x = makeHyperlinkString(
               sheet = "C-F1",
               row = 1,
               col = 1,
               text = "Valor estimado del financiamiento de la edificación, por trimestres, según tipo de obra"))

"C-F2"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 22,
             x = makeHyperlinkString(
               sheet = "C-F2",
               row = 1,
               col = 1,
               text = "Valor estimado del financiamiento de la edificación, por trimestres, según uso de la edificación"))

"C-F3"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 23,
             x = makeHyperlinkString(
               sheet = "C-F3",
               row = 1,
               col = 1,
               text = "Valor estimado de la edificación, por trimestres, según cantones"))


"Reaplicación de diseño"
addStyle(tab_ESED,
         "ÍNDICE",
         fill_a1,
         cols = 3,
         rows = c(6, 8, 10:13, 15:17, 19, 21:23),
         gridExpand = T)


# ============================================================================ #
##################   Creación de hipervínculos en cada hoja   ##################
# ============================================================================ #

"Diseño de la celda del hipervínculo"
hyp_ind <- createStyle(fontName = "Century Gothic",
                       fontSize = 15,
                       fontColour = "#FFFFFF",
                       textDecoration = "bold",
                       fgFill = "#7F7F7F",
                       halign = "center",
                       valign = "center",
                       wrapText = T)

"C-A1"
insertImage(tab_ESED,
            "C-A1",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\flecha.png",
            height = 2.15,
            width = 2.12,
            startCol = 9,
            startRow = 4,
            units = "cm")

writeFormula(tab_ESED,
             "C-A1",
             startCol = 10,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "C-A1",
         hyp_ind,
         cols = 10,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "C-A1",
             cols = 10,
             widths = "15")

"C-B1"
insertImage(tab_ESED,
            "C-B1",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\flecha.png",
            height = 2.15,
            width = 2.12,
            startCol = 18,
            startRow = 4,
            units = "cm")

writeFormula(tab_ESED,
             "C-B1",
             startCol = 19,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "C-B1",
         hyp_ind,
         cols = 19,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "C-B1",
             cols = 19,
             widths = "15")

"C-C1"
insertImage(tab_ESED,
            "C-C1",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\flecha.png",
            height = 2.15,
            width = 2.12,
            startCol = 18,
            startRow = 4,
            units = "cm")

writeFormula(tab_ESED,
             "C-C1",
             startCol = 19,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "C-C1",
         hyp_ind,
         cols = 19,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "C-C1",
             cols = 19,
             widths = "15")

"C-C2"
insertImage(tab_ESED,
            "C-C2",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\flecha.png",
            height = 2.15,
            width = 2.12,
            startCol = 7,
            startRow = 4,
            units = "cm")

mergeCells(tab_ESED,
           "C-C2",
           cols = 8,
           rows = 4:5)

writeFormula(tab_ESED,
             "C-C2",
             startCol = 8,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "C-C2",
         hyp_ind,
         cols = 8,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "C-C2",
             cols = 8,
             widths = "15")

"C-C3"
insertImage(tab_ESED,
            "C-C3",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\flecha.png",
            height = 2.56,
            width = 2.12,
            startCol = 18,
            startRow = 4,
            units = "cm")

writeFormula(tab_ESED,
             "C-C3",
             startCol = 19,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "C-C3",
         hyp_ind,
         cols = 19,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "C-C3",
             cols = 19,
             widths = "15")

"C-C4"
insertImage(tab_ESED,
            "C-C4",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\flecha.png",
            height = 2.15,
            width = 2.12,
            startCol = 7,
            startRow = 4,
            units = "cm")

mergeCells(tab_ESED,
           "C-C4",
           cols = 8,
           rows = 4:5)

writeFormula(tab_ESED,
             "C-C4",
             startCol = 8,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "C-C4",
         hyp_ind,
         cols = 8,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "C-C4",
             cols = 8,
             widths = "15")

"C-D1"
insertImage(tab_ESED,
            "C-D1",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\flecha.png",
            height = 2.15,
            width = 2.12,
            startCol = 18,
            startRow = 4,
            units = "cm")

writeFormula(tab_ESED,
             "C-D1",
             startCol = 19,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "C-D1",
         hyp_ind,
         cols = 19,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "C-D1",
             cols = 19,
             widths = "15")

"C-D2"
insertImage(tab_ESED,
            "C-D2",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\flecha.png",
            height = 2.15,
            width = 2.12,
            startCol = 10,
            startRow = 4,
            units = "cm")

mergeCells(tab_ESED,
           "C-D2",
           cols = 11,
           rows = 4:5)

writeFormula(tab_ESED,
             "C-D2",
             startCol = 11,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "C-D2",
         hyp_ind,
         cols = 11,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "C-D2",
             cols = 11,
             widths = "15")

"C-D3"
insertImage(tab_ESED,
            "C-D3",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\flecha.png",
            height = 2.15,
            width = 2.12,
            startCol = 8,
            startRow = 4,
            units = "cm")

mergeCells(tab_ESED,
           "C-D3",
           cols = 9,
           rows = 4:5)

writeFormula(tab_ESED,
             "C-D3",
             startCol = 9,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "C-D3",
         hyp_ind,
         cols = 9,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "C-D3",
             cols = 9,
             widths = "15")

"C-E1"
insertImage(tab_ESED,
            "C-E1",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\flecha.png",
            height = 2.15,
            width = 2.12,
            startCol = 9,
            startRow = 4,
            units = "cm")

mergeCells(tab_ESED,
           "C-E1",
           cols = 10,
           rows = 4:5)

writeFormula(tab_ESED,
             "C-E1",
             startCol = 10,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "C-E1",
         hyp_ind,
         cols = 10,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "C-E1",
             cols = 10,
             widths = "15")

"C-F1"
insertImage(tab_ESED,
            "C-F1",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\flecha.png",
            height = 2.15,
            width = 2.12,
            startCol = 7,
            startRow = 4,
            units = "cm")

mergeCells(tab_ESED,
           "C-F1",
           cols = 8,
           rows = 4:5)

writeFormula(tab_ESED,
             "C-F1",
             startCol = 8,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "C-F1",
         hyp_ind,
         cols = 8,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "C-F1",
             cols = 8,
             widths = "15")

"C-F2"
insertImage(tab_ESED,
            "C-F2",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\flecha.png",
            height = 2.56,
            width = 2.12,
            startCol = 19,
            startRow = 4,
            units = "cm")

writeFormula(tab_ESED,
             "C-F2",
             startCol = 20,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "C-F2",
         hyp_ind,
         cols = 20,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "C-F2",
             cols = 20,
             widths = "15")

"C-F3"
insertImage(tab_ESED,
            "C-F3",
            "C:\\Tabulados ESED IVT 2023\\Títulos tabulados\\flecha.png",
            height = 2.15,
            width = 2.12,
            startCol = 18,
            startRow = 4,
            units = "cm")

writeFormula(tab_ESED,
             "C-F3",
             startCol = 19,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "C-F3",
         hyp_ind,
         cols = 19,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "C-F3",
             cols = 19,
             widths = "15")


# ============================================================================ #
####################   Mejoras finales al diseño del Excel   ###################
# ============================================================================ #

showGridLines(tab_ESED, "ÍNDICE", showGridLines = F)
showGridLines(tab_ESED, "C-A1", showGridLines = F)
showGridLines(tab_ESED, "C-B1", showGridLines = F)
showGridLines(tab_ESED, "C-C1", showGridLines = F)
showGridLines(tab_ESED, "C-C2", showGridLines = F)
showGridLines(tab_ESED, "C-C3", showGridLines = F)
showGridLines(tab_ESED, "C-C4", showGridLines = F)
showGridLines(tab_ESED, "C-D1", showGridLines = F)
showGridLines(tab_ESED, "C-D2", showGridLines = F)
showGridLines(tab_ESED, "C-D3", showGridLines = F)
showGridLines(tab_ESED, "C-E1", showGridLines = F)
showGridLines(tab_ESED, "C-F1", showGridLines = F)
showGridLines(tab_ESED, "C-F2", showGridLines = F)
showGridLines(tab_ESED, "C-F3", showGridLines = F)

# ============================================================================ #
#####################   Exportación de tabulados a Excel   #####################
# ============================================================================ #

saveWorkbook(tab_ESED, "5. 2023_ESED_Tabulados-IVT.xlsx", overwrite = T)

#####                           FIN SCRIPT                             #####
