#==============================================================================#
#####     ESTADÍSTICAS DE EDIFICACIONES (PERMISOS DE CONSTRUCCIÓN) 2025     ####
#==============================================================================#
#==============================================================================#
## GENERACIÓN DE TABULADOS - ESTADÍSTICAS DE EDIFICACIONES III TRIMESTRE 2025  ##
#==============================================================================#
#==============================================================================#
#####           INSTITUTO NACIONAL DE ESTADÍSTICA Y CENSOS (INEC)          ####*
#==============================================================================#
# FUENTE: Estadísticas de Edificaciones 2022 - 2025.
# RESPONSABLE: Gestión de Estadísticas Estructurales (GESE).
# FECHA DE ELABORACIÓN: Noviembre 2025.
# FECHA ÚLTIMA DE ACTUALIZACIÓN: 21/11/2025.
# ==============================================================================*

# ==============================================================================*
# Elaborado por:
# Unidad de Gestión de Estadísticas Estructurales (GESE).
# Valery Paz y Miño.
# Enrique Vallejo.
# ==============================================================================*
# Revisado por:
# Lorena Ramos.
# Responsable de Estadísticas de Edificaciones (ESED).
# Roberto Chaves.
# Responsable de Gestión de Estadísticas Estructurales (GESE).
# ==============================================================================*
# Aprobado por: 
# Diana Barco.
# Directora de Estadísticas Económicas (DECON).
# ==============================================================================*

# ==============================================================================*
######                          INSTRUCCIONES                              #####*
# ==============================================================================*
# 1.- Descargar las bases de datos de las Estadísticas de Edificaciones de los años 
# 2022, 2023, 2024 en formato SPSS de la página web oficial del INEC:

# https://www.ecuadorencifras.gob.ec/edificaciones/ 

# Adicional, la base de datos del III trimestre del 2025  en formato SPSS de la página web
# oficial del INEC:

# (https://www.ecuadorencifras.gob.ec/esed-encuesta-edificaciones-trimestral/)

# 2.- En el DISCO C de su computadora, crear una carpeta llamada "Tabulados ESED IIIT 2025". 
# En esta carpeta  guardar las bases de datos históricas y la carpeta llamada "Títulos tabulados",
# aquí se exportará todos los resultados en Excel (Tabulados).
# =============================================================================*

####      ************************ NOTA ******************************    #### *
# En este SCRIPT, se encuentra toda la programación necesaria para la generación
# de los diferentes tabulados, mismos que fueron realizados en R-STUDIO 4.2.2. 
# **************************************************************************** #

# =============================================================================*
#                            LIMPIEZA DE CONSOLA                             ###
# =============================================================================*

rm(list=ls())  # Limpiar las ventanas de trabajo. 
cat("\014")

#==============================================================================#
#                              FUNCIONES                                       #
#==============================================================================#

"Cargar librerias y en caso de no existir, las instale"

esed_lib <- function(libreria) {
  if (!requireNamespace(as.character(libreria), quietly = TRUE)) {
    install.packages(as.character(libreria))
  }
  library(libreria, character.only = TRUE)
}

#==============================================================================#
###                      CARGA DE LIBRERÍAS                                ####*
#==============================================================================#

esed_lib("tidyverse")
esed_lib("openxlsx")
esed_lib("dplyr")
esed_lib("haven")

#==============================================================================#
###                          RUTA DE TRABAJO                                ####
#==============================================================================#

getwd() # Ver el directorio actual de trabajo (carpeta donde está la documentación - DISCO C) 
setwd("C:/Tabulados ESED IIIT 2025") # Directorio de trabajo del III trimestre 2025 (ver instrucción 2). 
dir()   # Ver los documentos que reposan en el directorio de trabajo establecido - DISCO C. 

#==============================================================================#
#####               IMPORTACIÓN DE LAS BASES DE DATOS                      #####
#==============================================================================#

# Con las bases descargadas de la página oficial del INEC, y colocadas en la ruta 
# de trabajo establecida. Se procede a la unificación de las bases de datos. (Favor no 
# cambiar el nombre de ninguna de las bases de datos).

db22 <- read_sav("6. 2022_ESED_BDD.sav") # Base de datos del año 2022
db23 <- read_sav("6. 2023_ESED_BDD.sav") # Base de datos del año 2023 
db24 <- read_sav("6. 2024_ESED_BDD.sav") # Base de datos del año 2024
db25_III <- read_sav("6. 2025_ESED_BDD_IIIT.sav") # Base de datos actual - III trimestre 2025

# Base de datos conjunta, más el III trimestre del 2025. 
ESED_2025_III <- bind_rows(db22, db23, db24, db25_III) # Base general

# Base auxiliar. 
dat <- ESED_2025_III  # Se utilizará esta BDD para la generación de nuevas categorías y variables.

rm(db22, db23, db24,db25_III)

#===================================================================================#
# Para la correcta generación de los tabulados, se deberá ejecutar todas las líneas #
# de código de forma completa (Ctrl + A). Si hubiese un error en la generación      #
# ejecutar (->RUN) nuevamente el código de forma completa.                          #
#===================================================================================#

# ============================================================================ #
#####               CREACIÓN DE VARIABLES PARA LOS TABULADOS               ####*
# ============================================================================ #

dat <- dat %>% 
  mutate(t = 1) %>% 
  mutate(tg = 1) 

attr(dat$t, "label") <- "TOTAL"
attr(dat$tg, "label") <- "TOTAL NACIONAL"

ESED_2025_III$t <- dat$t
ESED_2025_III$tg <- dat$tg

# ============================================================================ #
#####               RECODIFICACIÓN DE VARIABLES PARA TABULADOS             #####
# ============================================================================ #

## Uso de la edificación (CDISPUSO), cruzada con Propiedad de la edificación (propie)
# para desagregar entre; Educación Pública y Privada, y Hospitales Públicos y Privados

dat <- dat %>% 
  mutate(CDISPUSX = case_when(CDISPUSO == 1 ~ 1, # Edificaciones con una vivienda
                              CDISPUSO == 2 ~ 2, # Edificaciones con dos viviendas
                              CDISPUSO == 3 ~ 3, # Edificaciones con tres o más viviendas
                              CDISPUSO == 4 ~ 4, # Comercial
                              CDISPUSO == 5 ~ 5, # Industrial
                              CDISPUSO == 6 ~ 6, # Edificio Administrativo
                              CDISPUSO == 7 & propie == 1 ~ 8, # Educación Privada
                              CDISPUSO == 7 & propie == 2 ~ 9, # Educación Pública
                              CDISPUSO == 8 ~ 10, # Cultura
                              CDISPUSO == 9 ~ 11, # Recreación Deporte
                              CDISPUSO == 10 & propie == 1 ~ 13, # Hospitales Clínicas Estab de Salud Privadas
                              CDISPUSO == 10 & propie == 2 ~ 14, # # Hospitales Clínicas Estab de Salud Privada
                              CDISPUSO == 11 ~ 15, # Transporte y Comunicaciones
                              CDISPUSO == 13 ~ 17, # Mixto Residencial y No Residencial
                              CDISPUSO == 14 ~ 18, # Otras
                              T ~ 0
                              )) %>%
  mutate(CDISPUSX = factor(CDISPUSX, 
                       levels = c(0, 1, 2, 3, 4, 5, 6, 8, 9,
                                  10, 11, 13, 14, 15, 17, 18),
                       labels = c("No Aplica",
                                  "Edificaciones con una vivienda",
                                  "Edificaciones con dos viviendas",
                                  "Edificaciones con tres o más viviendas",
                                  "Comercial",
                                  "Industrial",
                                  "Edificio Administrativo",
                                  "Educación Privada",
                                  "Educación Pública",
                                  "Cultura",
                                  "Recreación Deporte",
                                  "Hospitales, Clínicas Estab de Salud Privada",
                                  "Hospitales, Clínicas Estab de Salud Pública",
                                  "Transporte y Comunicaciones",
                                  "Mixto Residencial y No Residencial",
                                  "Otras")))

attr(dat$CDISPUSX, "label") <- "PROPÓSITO DE LA CONSTRUCCIÓN"

ESED_2025_III$CDISPUSX <- dat$CDISPUSX

###### Se crea la variable total de área a construir sin parqueadero ####*

dat$CARCSING <- dat$CARCO - dat$CARDPAR

attr(dat$CARCSING, "label") <- "AREA TOTAL SIN PARQUEADERO"

ESED_2025_III$CARCSING <- dat$CARCSING

###### Número de pisos para edificaciones de tipo residencial ######*

dat <- dat %>% 
  mutate(TCDISREP = case_when(CDISPUSO >= 1 & CDISPUSO <= 3 ~ 1, T ~ 0)) %>% 
  mutate(CDISREP = case_when(CDISPUSO >= 1 & CDISPUSO <= 3 & NUPICAL == 1 ~ 1,
                             CDISPUSO >= 1 & CDISPUSO <= 3 & NUPICAL == 2 ~ 2,
                             CDISPUSO >= 1 & CDISPUSO <= 3 & NUPICAL == 3 ~ 3,
                             CDISPUSO >= 1 & CDISPUSO <= 3 & NUPICAL >= 4 ~ 4,
                             T ~ 0
                             )) %>% 
  mutate(CDISREP = factor(CDISREP, levels = c(1:4),
                          labels = c("1 PISO",
                                     "2 PISOS",
                                     "3 PISOS",
                                     "4 Y MÁS PISOS"
                                     )))

attr(dat$CDISREP, "label") <- "RESIDENCIAL"
attr(dat$TCDISREP, "label") <- "SUBTOTAL RESIDENCIAL" 

ESED_2025_III$TCDISREP <- dat$TCDISREP
ESED_2025_III$CDISREP <- dat$CDISREP

###### Número de pisos para edificaciones de tipo No Residencial ######*

dat <- dat %>% 
  mutate(TCDISNRP = case_when(((CDISPUSO >= 4 & CDISPUSO <= 12) | CDISPUSO == 14 ~ 1))) %>% 
  mutate(CDISNRP = case_when((CDISPUSO >= 4 & CDISPUSO <= 12 | CDISPUSO == 14) & NUPICAL == 1 ~ 1,
                             (CDISPUSO >= 4 & CDISPUSO <= 12 | CDISPUSO == 14) & NUPICAL == 2 ~ 2,
                             (CDISPUSO >= 4 & CDISPUSO <= 12 | CDISPUSO == 14) & NUPICAL == 3 ~ 3,
                             (CDISPUSO >= 4 & CDISPUSO <= 12 | CDISPUSO == 14) & NUPICAL >= 4 ~ 4,
                             T ~ 0
                             )) %>% 
  mutate(CDISNRP = factor(CDISNRP, levels = c(1:4),
                           labels = c("1 PISO",
                                      "2 PISOS",
                                      "3 PISOS",
                                      "4 Y MÁS PISOS"
                                      )))

attr(dat$TCDISNRP, "label")  <- "SUBTOTAL NO RESIDENCIAL"
attr(dat$CDISNRP, "label")  <- "NO RESIDENCIAL"

ESED_2025_III$TCDISNRP <- dat$TCDISNRP
ESED_2025_III$CDISNRP <- dat$CDISNRP

###### Número de pisos para edificaciones de tipo Mixta ######*

dat <- dat %>% 
  mutate(TCDISMIP = ifelse((CDISPUSO == 13), 1, NA)) %>% 
  mutate(CDISMIP = case_when(CDISPUSO == 13 & NUPICAL == 1 ~ 1,
                             CDISPUSO == 13 & NUPICAL == 2 ~ 2,
                             CDISPUSO == 13 & NUPICAL == 3 ~ 3,
                             CDISPUSO == 13 & NUPICAL >= 4 ~ 4
                             )) %>% 
  mutate(CDISMIP = factor(CDISMIP, levels = c(1:4),
                           labels = c("1 PISO",
                                      "2 PISOS",
                                      "3 PISOS",
                                      "4 Y MÁS PISOS"
                                      )))

attr(dat$TCDISMIP, "label")  <- "SUBTOTAL MIXTA"
attr(dat$CDISMIP, "label") <- "MIXTA"

ESED_2025_III$TCDISMIP <- dat$TCDISMIP
ESED_2025_III$CDISMIP <- dat$CDISMIP

###### Número de cuartos para Residencial de 1 Familia ######*

dat <- dat %>% 
  mutate(TCVIVREP = ifelse((CDISPUSO == 1), 1, NA)) %>% 
  mutate(CVIVREP = case_when(CDISPUSO == 1 & NUCUCAL == 1 ~ 1,
                             CDISPUSO == 1 & NUCUCAL == 2 ~ 2,
                             CDISPUSO == 1 & NUCUCAL == 3 ~ 3,
                             CDISPUSO == 1 & NUCUCAL == 4 ~ 4,
                             CDISPUSO == 1 & NUCUCAL == 5 ~ 5,
                             CDISPUSO == 1 & NUCUCAL == 6 ~ 6,
                             CDISPUSO == 1 & NUCUCAL >= 7 ~ 7
                             )) %>% 
  mutate(CVIVREP = factor(CVIVREP, levels = c(1:7),
                           labels = c("1 CUARTO",
                                      "2 CUARTOS",
                                      "3 CUARTOS",
                                      "4 CUARTOS",
                                      "5 CUARTOS",
                                      "6 CUARTOS",
                                      "7 Y MÁS CUARTOS"
                                      )))

attr(dat$TCVIVREP, "label") <- "SUBTOTAL 1 FAMILIA"
attr(dat$CVIVREP, "label") <- "1 FAMILIA"

ESED_2025_III$TCVIVREP <- dat$TCVIVREP
ESED_2025_III$CVIVREP <- dat$CVIVREP

###### Número de cuartos para Residencial de 2 Familias ###### *

dat <- dat %>% 
  mutate(TCVIVRE2 = ifelse(CDISPUSO == 2, 1, NA)) %>% 
  mutate(CVIVRE2 = case_when(CDISPUSO == 2 & NUCUCAL == 1 ~ 1,
                             CDISPUSO == 2 & NUCUCAL == 2 ~ 2,
                             CDISPUSO == 2 & NUCUCAL == 3 ~ 3,
                             CDISPUSO == 2 & NUCUCAL == 4 ~ 4,
                             CDISPUSO == 2 & NUCUCAL == 5 ~ 5,
                             CDISPUSO == 2 & NUCUCAL == 6 ~ 6,
                             CDISPUSO == 2 & NUCUCAL >= 7 ~ 7
                             )) %>% 
  mutate(CVIVRE2 = factor(CVIVRE2, levels = c(1:7),
                          labels = c("1 CUARTO",
                                     "2 CUARTOS",
                                     "3 CUARTOS",
                                     "4 CUARTOS",
                                     "5 CUARTOS",
                                     "6 CUARTOS",
                                     "7 Y MÁS CUARTOS"
                                     )))

attr(dat$TCVIVRE2, "label") <- "SUBTOTAL 2 FAMILIAS"
attr(dat$CVIVRE2, "label") <- "2 FAMILIAS"

ESED_2025_III$TCVIVRE2 <- dat$TCVIVRE2
ESED_2025_III$CVIVRE2 <- dat$CVIVRE2

###### Número de cuartos para Residencial de 3 Familias ###### *

dat <- dat %>% 
  mutate(TCVIVRE3 = ifelse((CDISPUSO == 3),1,NA)) %>% 
  mutate(CVIVRE3 = case_when(CDISPUSO == 3 & NUCUCAL == 1 ~ 1,
                             CDISPUSO == 3 & NUCUCAL == 2 ~ 2,
                             CDISPUSO == 3 & NUCUCAL == 3 ~ 3,
                             CDISPUSO == 3 & NUCUCAL == 4 ~ 4,
                             CDISPUSO == 3 & NUCUCAL == 5 ~ 5,
                             CDISPUSO == 3 & NUCUCAL == 6 ~ 6,
                             CDISPUSO == 3 & NUCUCAL >= 7 ~ 7
                             )) %>% 
  mutate(CVIVRE3 = factor(CVIVRE3, levels = c(1:7),
                          labels = c("1 CUARTO",
                                     "2 CUARTOS",
                                     "3 CUARTOS",
                                     "4 CUARTOS",
                                     "5 CUARTOS",
                                     "6 CUARTOS",
                                     "7 Y MÁS CUARTOS"
                                     )))
         
attr(dat$TCVIVRE3, "label") <- "SUBTOTAL 3 FAMILIAS"
attr(dat$CVIVRE3, "label") <- "3 FAMILIAS"

ESED_2025_III$TCVIVRE3 <- dat$TCVIVRE3
ESED_2025_III$CVIVRE3 <- dat$CVIVRE3

###### Número de cuartos para Residencial Mixta ######*

dat <- dat %>% 
  mutate(TCVIVRE4 = ifelse((CDISPUSO == 13), 1, NA)) %>% 
  mutate(CVIVRE4 = case_when(CDISPUSO == 13 & NUCUCAL == 1 ~ 1,
                             CDISPUSO == 13 & NUCUCAL == 2 ~ 2,
                             CDISPUSO == 13 & NUCUCAL == 3 ~ 3,
                             CDISPUSO == 13 & NUCUCAL == 4 ~ 4,
                             CDISPUSO == 13 & NUCUCAL == 5 ~ 5,
                             CDISPUSO == 13 & NUCUCAL == 6 ~ 6,
                             CDISPUSO == 13 & NUCUCAL >= 7 ~ 7
                             )) %>% 
  mutate(CVIVRE4 = factor(CVIVRE4, levels = c(1:7),
                          labels = c("1 CUARTO",
                                     "2 CUARTOS",
                                     "3 CUARTOS",
                                     "4 CUARTOS",
                                     "5 CUARTOS",
                                     "6 CUARTOS",
                                     "7 Y MÁS CUARTOS"
                                     )))

attr(dat$TCVIVRE4, "label") <- "SUBTOTAL MIXTA"
attr(dat$CVIVRE4, "label") <- "MIXTA"

ESED_2025_III$TCVIVRE4 <- dat$TCVIVRE4
ESED_2025_III$CVIVRE4 <- dat$CVIVRE4


###### Número de dormitorios para  Residencial 1 Familia ###### *

dat <- dat %>% 
  mutate(TDVIVREP = ifelse((CDISPUSO == 1), 1, NA)) %>% 
  mutate(DVIVREP = case_when(CDISPUSO == 1 & NUDOCAL == 1 ~ 1,
                             CDISPUSO == 1 & NUDOCAL == 2 ~ 2,
                             CDISPUSO == 1 & NUDOCAL == 3 ~ 3,
                             CDISPUSO == 1 & NUDOCAL == 4 ~ 4,
                             CDISPUSO == 1 & NUDOCAL == 5 ~ 5,
                             CDISPUSO == 1 & NUDOCAL == 6 ~ 6,
                             CDISPUSO == 1 & NUDOCAL >= 7 ~ 7
                             )) %>% 
  mutate(DVIVREP = factor(DVIVREP, levels = c(1:7),
                          labels = c("1 DORMITORIO",
                                     "2 DORMITORIOS",
                                     "3 DORMITORIOS",
                                     "4 DORMITORIOS",
                                     "5 DORMITORIOS",
                                     "6 DORMITORIOS",
                                     "7 Y MÁS DORMITORIOS"
                                     )))

attr(dat$TDVIVREP, "label") <- "SUBTOTAL 1 FAMILIA"
attr(dat$DVIVREP, "label") <- "1 FAMILIA"

ESED_2025_III$TDVIVREP <- dat$TDVIVREP
ESED_2025_III$DVIVREP <- dat$DVIVREP


###### Número de dormitorios para Residencial 2 Familias ###### *

dat <- dat %>% 
  mutate(TDVIVRE2 = ifelse((CDISPUSO == 2), 1, NA)) %>% 
  mutate(DVIVRE2 = case_when(CDISPUSO == 2 & NUDOCAL == 1 ~ 1,
                             CDISPUSO == 2 & NUDOCAL == 2 ~ 2,
                             CDISPUSO == 2 & NUDOCAL == 3 ~ 3,
                             CDISPUSO == 2 & NUDOCAL == 4 ~ 4,
                             CDISPUSO == 2 & NUDOCAL == 5 ~ 5,
                             CDISPUSO == 2 & NUDOCAL == 6 ~ 6,
                             CDISPUSO == 2 & NUDOCAL >= 7 ~ 7
                             )) %>% 
  mutate(DVIVRE2 = factor(DVIVRE2, levels = c(1:7),
                          labels = c("1 DORMITORIO",
                                     "2 DORMITORIOS",
                                     "3 DORMITORIOS",
                                     "4 DORMITORIOS",
                                     "5 DORMITORIOS",
                                     "6 DORMITORIOS",
                                     "7 Y MÁS DORMITORIOS"
                                     )))

attr(dat$TDVIVRE2, "label") <- "SUBTOTAL 2 FAMILIAS"
attr(dat$DVIVRE2, "label") <- "2 FAMILIAS"

ESED_2025_III$TDVIVRE2 <- dat$TDVIVRE2
ESED_2025_III$DVIVRE2 <- dat$DVIVRE2


###### Número de dormitorios para Residencial 3 Familias ######*

dat <- dat %>% 
  mutate(TDVIVRE3 = ifelse((CDISPUSO == 3),1,NA)) %>% 
  mutate(DVIVRE3 = case_when(CDISPUSO == 3 & NUDOCAL == 1 ~ 1,
                             CDISPUSO == 3 & NUDOCAL == 2 ~ 2,
                             CDISPUSO == 3 & NUDOCAL == 3 ~ 3,
                             CDISPUSO == 3 & NUDOCAL == 4 ~ 4,
                             CDISPUSO == 3 & NUDOCAL == 5 ~ 5,
                             CDISPUSO == 3 & NUDOCAL == 6 ~ 6,
                             CDISPUSO == 3 & NUDOCAL >= 7 ~ 7
                             )) %>% 
  mutate(DVIVRE3 = factor(DVIVRE3, levels = c(1:7),
                          labels = c("1 DORMITORIO",
                                     "2 DORMITORIOS",
                                     "3 DORMITORIOS",
                                     "4 DORMITORIOS",
                                     "5 DORMITORIOS",
                                     "6 DORMITORIOS",
                                     "7 Y MÁS DORMITORIOS"
                                     )))

attr(dat$TDVIVRE3, "label") <- "SUBTOTAL 3 FAMILIAS"
attr(dat$DVIVRE3, "label") <- "3 FAMILIAS"

ESED_2025_III$TDVIVRE3 <- dat$TDVIVRE3
ESED_2025_III$DVIVRE3 <- dat$DVIVRE3

###### Número de dormitorios para Residencial Mixta ###### *

dat <- dat %>% 
  mutate(TDVIVRE4 = ifelse((CDISPUSO == 13), 1, NA)) %>% 
  mutate(DVIVRE4 = case_when(CDISPUSO == 13 & NUDOCAL == 1 ~ 1,
                             CDISPUSO == 13 & NUDOCAL == 2 ~ 2,
                             CDISPUSO == 13 & NUDOCAL == 3 ~ 3,
                             CDISPUSO == 13 & NUDOCAL == 4 ~ 4,
                             CDISPUSO == 13 & NUDOCAL == 5 ~ 5,
                             CDISPUSO == 13 & NUDOCAL == 6 ~ 6,
                             CDISPUSO == 13 & NUDOCAL >= 7 ~ 7
                             )) %>% 
  mutate(DVIVRE4 = factor(DVIVRE4, levels = c(1:7),
                          labels = c("1 DORMITORIO",
                                     "2 DORMITORIOS",
                                     "3 DORMITORIOS",
                                     "4 DORMITORIOS",
                                     "5 DORMITORIOS",
                                     "6 DORMITORIOS",
                                     "7 Y MÁS DORMITORIOS"
                                     )))

attr(dat$TDVIVRE4, "label") <- "SUBTOTAL MIXTA"
attr(dat$DVIVRE4, "label") <- "MIXTA"

ESED_2025_III$TDVIVRE4 <- dat$TDVIVRE4
ESED_2025_III$DVIVRE4 <- dat$DVIVRE4


###### Se recodifica la variable CIMIENTOS para que el "No aplica" se cargue a la categoría "otros" ###*

dat <- dat %>% 
  mutate(RCIMI = case_when(cimi == 120 ~ 120, # Hormigón armado
                           cimi == 124 ~ 124, # Hormigón ciclópeo
                           cimi == 147 ~ 147, # Cimiento portantes
                           cimi == 160 ~ 160, # Sobre pilotes
                           cimi == 136 ~ 136, # Otros
                           cimi == 0 ~ 136,
                           T ~ 0
                           )) %>% 
  mutate(RCIMI = factor(RCIMI, levels = c(120, 124, 147, 160, 136),
                        labels = c("HORMIGÓN ARMADO",
                                   "HORMIGÓN CICLÓPEO",
                                   "CIMIENTOS PORTANTES",
                                   "SOBRE PILOTES",
                                   "OTROS"
                                   )))

attr(dat$RCIMI, "label") <- "CIMIENTOS"

ESED_2025_III$RCIMI <- dat$RCIMI

###### Se recodifica la variable ESTRUCTURA para que el "No aplica" se cargue a la categoría "otros" ##*

dat <- dat %>% 
  mutate(RESTRU = case_when(estru == 121 ~ 121, # Hormigón armado
                            estru == 125 ~ 125, # Metalica
                            estru == 129 ~ 129, # Madera
                            estru == 148 ~ 148, # Muros portantes
                            estru == 133 ~ 133, #Otros
                            estru == 0 ~ 133,
                            T ~ 0
                            )) %>% 
  mutate(RESTRU = factor(RESTRU, levels = c(121, 125, 129, 148, 133),
                         labels = c("HORMIGÓN ARMADO",
                                    "METÁLICA",
                                    "MADERA",
                                    "MUROS PORTANTES",
                                    "OTROS"
                                    )))

attr(dat$RESTRU, "label") <- "ESTRUCTURA"

ESED_2025_III$RESTRU <- dat$RESTRU


###### Se recodifica la variable PARED para que el "No aplica" se cargue a la categoría "otros" #*

dat <- dat %>% 
  mutate(RPARED = case_when(pared == 122 ~ 122, # Ladrillo
                            pared == 126 ~ 126, # Bloque
                            pared == 130 ~ 130, # Madera
                            pared == 134 ~ 134, # Adobe o tapia
                            pared == 138 ~ 138, # Bahreque
                            pared == 142 ~ 142, # Prefabricadas
                            pared == 150 ~ 150, # Planchas de carton yeso
                            pared == 164 ~ 164, # Vidrio
                            pared == 165 ~ 165, # Hormigon
                            pared == 146 ~ 146, # Otros
                            pared == 0 ~ 146,
                            T ~ 0
                            )) %>% 
  mutate(RPARED = factor(RPARED, levels = c(122, 126, 130, 134, 138,
                                            142,150, 164, 165 ,146),
                         labels = c("Ladrillo",
                                    "Bloque",
                                    "Madera",
                                    "Adobe o tapia",
                                    "Bahreque",
                                    "Prefabricadas",
                                    "Planchas de carton yeso",
                                    "Vidrio",
                                    "Hormigon",
                                    "Otros"
                                    )))
attr(dat$RPARED, "label") <- "PARED"

ESED_2025_III$RPARED <- dat$RPARED

###### Se recodifica la variable CUBIERTA para que el "No aplica" se cargue a la categoría "otros" #*

dat <- dat %>% 
  mutate(RCUBI = case_when(cubi == 123 ~ 123, # Hormigón armado
                           cubi == 127 ~ 127, # Fibrocemento (eternit, ardex)
                           cubi == 131 ~ 131, # Arcilla (teja)
                           cubi == 135 ~ 135, # Láminas metálicas (zinc)
                           cubi == 139 ~ 139, # Policarbonato
                           cubi == 166 ~ 166, # Losa mixta
                           cubi == 167 ~ 167, # Madera y capa asfaltica
                           cubi == 168 ~ 168, # Tipo sanduche o metalico poliuretano
                           cubi == 143 ~ 143, # Otros
                           cubi == 0 ~ 143,
                           T ~ 0
                           )) %>% 
  mutate(RCUBI = factor(RCUBI, levels = c(123, 127, 131, 135, 139, 166, 167,168, 143),
                        labels = c("HORMIGÓN ARMADO",
                                   "Fibrocemento (eternit, ardex)",
                                   "Arcilla (teja)",
                                   "Láminas metálicas (zinc)",
                                   "Policarbonato",
                                   "Losa mixta",
                                   "Madera y capa asfaltica",
                                   "Tipo sanduche o metalico poliuretano",
                                   "OTROS"
                                   )))
attr(dat$RCUBI, "label") <- "CUBIERTA"


ESED_2025_III$RCUBI <- dat$RCUBI


###### Se recodifica la variable PISOS para que el "No aplica" se cargue a la categoría "otros" #*

dat <- dat %>% 
  mutate(RPISO = case_when(piso == 151 ~ 151, # Tierra estabilizada
                           piso == 152 ~ 152, # Hormigón
                           piso == 153 ~ 153, # Madera
                           piso == 154 ~ 154, # Cerámica
                           piso == 155 ~ 155, # Porcelanato
                           piso == 156 ~ 156, # Piso laminado(piso flotante
                           piso == 157 ~ 157, # Marmol
                           piso == 161 ~ 161, # Cemento pulido
                           piso == 162 ~ 162, # Piso epoxico
                           piso == 163 ~ 163, # Otros
                           piso == 0 ~ 157,
                           T ~ 0
                           )) %>% 
  mutate(RPISO = factor(RPISO, levels = c(151,152,153,154,155,156,157,161,162,163),
                        labels = c("TIERRA ESTABILIZADA",
                                   "HORMIGÓN",
                                   "MADERA",
                                   "CERÁMICA",
                                   "PORCELANATO",
                                   "PISO LAMINADO(PISO FLOTANTE)",
                                   "MÁRMOL",
                                   "CEMENTO PULIDO",
                                   "PESO EPÓXICO",
                                   "OTROS"
                                   )))

attr(dat$RPISO, "label") <- "PISOS"

ESED_2025_III$RPISO <- dat$RPISO

###### Se recodifica la variable ENTREPISOS para que el "No aplica" se cargue a la categoría "otros" #*

dat <- dat %>% 
  mutate(RENTREPISO = case_when(entrepisos == 1 ~ 1, # Hormigón armado
                                entrepisos == 2 ~ 2, # Metálica
                                entrepisos == 3 ~ 3, # Madera
                                entrepisos == 4 ~ 4, # Mixta (Hormigón/Metálica)
                                entrepisos == 5 ~ 5, #Otros
                                entrepisos == 0 ~ 5, 
                                T ~ 0
  )) %>% 
  mutate(RENTREPISO = factor(RENTREPISO, levels = c(1, 2, 3, 4, 5),
                         labels = c("HORMIGÓN ARMADO",
                                    "METÁLICA",
                                    "MADERA",
                                    "MIXTA (HORMIGÓN/METÁLICA",
                                    "OTROS"
                         )))

attr(dat$RENTREPISO, "label") <- "ENTREPISOS"

ESED_2025_III$RENTREPISO <- dat$RENTREPISO
  
#### Para los tabulados se debe crear la variable NÚMERO DE PERMISOS DE CONSTRUCCIÓN,
#### que es igual a la variable CPERM, pero en esta, se cuenta el número de registros ###### *

dat <- dat %>% 
  mutate(NUMPER = CPERM)

attr(dat$NUMPER, "label") <- "NÚMERO DE PERMISOS DE CONSTRUCCIÓN"

ESED_2025_III$NUMPER <- dat$NUMPER


#### Debemos generar un nueva variable: NÚMERO DE PISOS TOTAL, ya que está por 
### uso residencial, no residencial y Mixta ##*

dat$NUPISOS <- ifelse(!is.na(dat$CDISREP), dat$CDISREP,
                      ifelse(!is.na(dat$CDISNRP), dat$CDISNRP,
                             ifelse(!is.na(dat$CDISMIP), dat$CDISMIP, NA
                                    )))

attr(dat$NUPISOS, "label") <- "NUMERO DE PISOS GENERAL"

ESED_2025_III$NUPISOS <- dat$NUPISOS


#### Debemos generar una variable: TOTAL DEL NUMERO DE CUARTOS, ya que está desagregada por 
#### Edificaciones de 1 vivienda, Edificaciones de 2 viviendas, Edificaciones de 3 viviendas y mixtas*

dat$NUCUARTOS <- ifelse(!is.na(dat$CVIVREP), dat$CVIVREP,
                        ifelse(!is.na(dat$CVIVRE2), dat$CVIVRE2,
                               ifelse(!is.na(dat$CVIVRE3), dat$CVIVRE3,
                                      ifelse(!is.na(dat$CVIVRE4), dat$CVIVRE4, NA
                                             ))))

attr(dat$NUCUARTOS, "label") <- "NUMERO DE CUARTOS GENERAL"

ESED_2025_III$NUCUARTOS <- dat$NUCUARTOS


#### Debemos generar una variable TOTAL DEL NUMERO DE DORMITORIOS, ya que está desagregada 
## por Edificaciones de 1 vivienda, Edificaciones de 2 viviendas, Edificaciones de 3 viviendas y mixtas*

dat$NUDORMITORIOS <- ifelse(!is.na(dat$DVIVREP), dat$DVIVREP,
                            ifelse(!is.na(dat$DVIVRE2), dat$DVIVRE2,
                                   ifelse(!is.na(dat$DVIVRE3), dat$DVIVRE3,
                                          ifelse(!is.na(dat$DVIVRE4), dat$DVIVRE4, NA
                                                 ))))
                      
attr(dat$NUDORMITORIOS, "label") <- "NUMERO DE DORMITORIOS GENERAL"

ESED_2025_III$NUDORMITORIOS <- dat$NUDORMITORIOS

##### Se debe generar una variable FINAN que separe a los recursos propios y prestados ####* 

dat <- dat %>% 
  mutate(FINAN = case_when((CDISPRFI == 1 |  CDISPRFI == 2 |  CDISPRFI == 3 | 
                              CDISPRFI == 4 |  CDISPRFI == 5) ~ 1,
                           (CDISPRFI == 6 |  CDISPRFI == 7 |  CDISPRFI == 8 |  
                              CDISPRFI == 9 |  CDISPRFI == 10 |  CDISPRFI == 11 | 
                              CDISPRFI == 12 | CDISPRFI == 13) ~ 2,
                           T ~ 0
                           )) %>% 
  mutate(FINAN = factor(FINAN, levels = c(0, 1, 2),
                        labels = c("No Aplica",
                                   "Recursos Propios",
                                   "Recursos Prestados"
                                   )))
attr(dat$FINAN, "label") <- "FINANCIAMIENTO PROPIO O PRESTADO"

ESED_2025_III$FINAN <- dat$FINAN

#### Se crea una variable llamada "TRIMES"  para obtener los trimestres por cada año (2021-2025) #*

dat <- dat %>% 
  mutate(TRIMES = case_when(anio == 2021 & (mes == 1 |  mes == 2 | mes == 3) ~ 1,
                            anio == 2021 & (mes == 4 |  mes == 5 | mes == 6) ~ 2,
                            anio == 2021 & (mes == 7 |  mes == 8 | mes == 9) ~ 3,
                            anio == 2021 & (mes == 10 |  mes == 11 | mes == 12) ~ 4,
                            anio == 2022 & (mes == 1 | mes == 2 | mes == 3) ~ 5,
                            anio == 2022 & (mes == 4 | mes == 5 | mes == 6) ~ 6,
                            anio == 2022 & (mes == 7 |  mes == 8 | mes == 9) ~ 7,
                            anio == 2022 & (mes == 10 |  mes == 11 | mes == 12) ~ 8,
                            anio == 2023 & (mes == 1 | mes == 2 | mes == 3) ~ 9,
                            anio == 2023 & (mes == 4 | mes == 5 | mes == 6) ~ 10,
                            anio == 2023 & (mes == 7 | mes == 8 | mes == 9) ~ 11,
                            anio == 2023 & (mes == 10 | mes == 11 | mes == 12) ~ 12,
                            anio == 2024 & (mes == 1 | mes == 2 | mes == 3) ~ 13,
                            anio == 2024 & (mes == 4 | mes == 5 | mes == 6) ~ 14,
                            anio == 2024 & (mes == 7 | mes == 8 | mes == 9) ~ 15,
                            anio == 2024 & (mes == 10 | mes == 11 | mes == 12) ~ 16,
                            anio == 2025 & (mes == 1 | mes == 2 | mes == 3) ~ 17,
                            anio == 2025 & (mes == 4 | mes == 5 | mes == 6) ~ 18,
                            anio == 2025 & (mes == 7 | mes == 8 | mes == 9) ~ 19,
                            T ~ 0
                            )) %>% 
  mutate(TRIMES = factor(TRIMES, levels = c(0:19),
                         labels = c("No Aplica",
                                    "2021-I",
                                    "2021-II",
                                    "2021-III",
                                    "2021-IV",
                                    "2022-I",
                                    "2022-II",
                                    "2022-III",
                                    "2022-IV", 
                                    "2023-I",
                                    "2023-II",
                                    "2023-III",
                                    "2023-IV",
                                    "2024-I" ,
                                    "2024-II",
                                    "2024-III",
                                    "2024-IV",
                                    "2025-I",
                                    "2025-II",
                                    "2025-III")))

attr(dat$TRIMES, "label") <- "TRIMESTRES POR AÑOS 2021 AL 2025"

ESED_2025_III$TRIMES <- dat$TRIMES

#### Se filtra por los 15 GAD municipales seleccionados #####*

ESED_2025_III <- ESED_2025_III %>% 
  filter(  codcantf == "1701" | codcantf == "0906" | codcantf == "0901" | 
           codcantf == "1801" | codcantf == "0101" | codcantf == "1001" | 
           codcantf == "1101" | codcantf == "0601" | codcantf == "2301" |
           codcantf == "1301" | codcantf == "0916" | codcantf == "1308" |
           codcantf == "0701" | codcantf == "0907" | codcantf == "2402" )


# ============================================================================ #
#####                TABULADOS DE LOS PRINCIPALES RESULTADOS               #####
# ============================================================================ #

# Se define la coma "," como separador de decimales
options(OutDec = ",")

###### A Cuadro Resumen ###### *

####### A1 Resultados de las principales variables #######

# 2022
CA1_2022 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2022-I", 
                       "2022-II", 
                       "2022-III", 
                       "2022-IV")) %>% 
  group_by(TRIMES) %>% 
  summarise(num_per = sum(tg, na.rm = T),
            num_ed = sum(CPERM, na.rm = T),
            num_viv = sum(CNUVICAL, na.rm = T),
            sup = sum(CSUTE, na.rm = T),
            area = sum(CARCO, na.rm = T),
            valor = sum(CVAE, na.rm = T))%>%
  mutate_at(c(2:7), ~replace(., is.na(.), 0)) %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CA1_2022)[5] <- "2022"

# 2023
CA1_2023 <- ESED_2025_III %>%
  filter(TRIMES %in% c("2023-I" , 
                       "2023-II",
                       "2023-III",
                       "2023-IV")) %>% 
  group_by(TRIMES) %>% 
  summarise(num_per = sum(tg, na.rm = T),
            num_ed = sum(CPERM, na.rm = T),
            num_viv = sum(CNUVICAL, na.rm = T),
            sup = sum(CSUTE, na.rm = T),
            area = sum(CARCO, na.rm = T),
            valor = sum(CVAE, na.rm = T)) %>%
  mutate_at(c(2:7), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CA1_2023)[5] <- "2023"

# 2024
CA1_2024 <- ESED_2025_III %>%
  filter(TRIMES %in% c("2024-I", 
                       "2024-II",
                       "2024-III",
                       "2024-IV")) %>% 
  group_by(TRIMES) %>% 
  summarise(num_per = sum(tg, na.rm = T),
            num_ed = sum(CPERM, na.rm = T),
            num_viv = sum(CNUVICAL, na.rm = T),
            sup = sum(CSUTE, na.rm = T),
            area = sum(CARCO, na.rm = T),
            valor = sum(CVAE, na.rm = T)) %>%
  mutate_at(c(2:7), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CA1_2024)[5] <- "2024"

# 2025
CA1_2025 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2025-I", 
                       "2025-II",
                       "2025-III")) %>% 
  group_by(TRIMES) %>% 
  summarise(num_per = sum(tg, na.rm = T),
            num_ed = sum(CPERM, na.rm = T),
            num_viv = sum(CNUVICAL, na.rm = T),
            sup = sum(CSUTE, na.rm = T),
            area = sum(CARCO, na.rm = T),
            valor = sum(CVAE, na.rm = T))%>%
  mutate_at(c(2:7), ~replace(., is.na(.), 0))%>%
  column_to_rownames(var = "TRIMES")


# Completo
CA1 <- bind_rows(CA1_2022, CA1_2023 , CA1_2024, CA1_2025)
names(CA1) <- c("Número de permisos de construcción",
                 "Número de edificaciones",
                 "Número de viviendas",
                 "Superficie del terreno (m2)",
                 "Área total a construir (m2)",
                 "Valor estimado de la edificación (dólares)")

rm(CA1_2022, CA1_2023,CA1_2024, CA1_2025)

###### B Número de permisos de construcción ######*

##### B1 Permisos de construcción, por trimestres, según cantones ####

# 2022
CB1_2022 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2022-I", 
                       "2022-II", 
                       "2022-III", 
                       "2022-IV")) %>% 
  select(TRIMES, codcantf, tg) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(tg, na.rm = T)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de permisos" = sbtotal) %>% 
    pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>% 
  rename("Quito" = "Distrito Metropolitano De Quito") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CB1_2022)[5] <- "2022"

# 2023
CB1_2023 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2023-I" ,
                       "2023-II",
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, codcantf, tg) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(tg, na.rm = T)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de permisos" = sbtotal) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>%
  rename("Quito" = "Distrito Metropolitano De Quito") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup()%>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CB1_2023)[5] <- "2023"

# 2024

CB1_2024 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2024-I",
                       "2024-II",
                       "2024-III",
                       "2024-IV")) %>% 
  select(TRIMES, codcantf, tg) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(tg, na.rm = T)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de permisos" = sbtotal) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>%
  rename("Quito" = "Distrito Metropolitano De Quito") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup()%>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CB1_2024)[5] <- "2024"

# 2025
CB1_2025 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2025-I", 
                       "2025-II",
                       "2025-III")) %>% 
  select(TRIMES, codcantf, tg) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(tg, na.rm = T)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de permisos" = sbtotal) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:15), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  select(sort(names(.)))


# Completo
CB1 <- bind_rows(CB1_2022, CB1_2023, CB1_2024, CB1_2025 ) %>% 
  mutate_all(~ifelse(is.na(.), 0, .))

rm(CB1_2022, CB1_2023, CB1_2024, CB1_2025)

###### C Número de edificaciones a construir ######*

####### C1 Edificaciones a construir, por trimestres, según cantones #######

# 2022
CC1_2022 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2022-I", 
                       "2022-II", 
                       "2022-III", 
                       "2022-IV")) %>% 
  select(TRIMES, codcantf, CPERM) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de edificaciones" = sbtotal) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>% 
  rename("Quito" = "Distrito Metropolitano De Quito") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CC1_2022)[5] <- "2022"

# 2023
CC1_2023 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2023-I",
                       "2023-II", 
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, codcantf, CPERM) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>%
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de edificaciones" = sbtotal) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>% 
  rename("Quito" = "Distrito Metropolitano De Quito") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup()%>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CC1_2023)[5] <- "2023"

# 2024
CC1_2024 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2024-I",
                       "2024-II",
                       "2024-III",
                       "2024-IV")) %>% 
  select(TRIMES, codcantf, CPERM) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de edificaciones" = sbtotal) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>% 
  rename("Quito" = "Distrito Metropolitano De Quito") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup()%>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CC1_2024)[5] <- "2024"

# 2025
CC1_2025 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2025-I", 
                       "2025-II",
                       "2025-III")) %>% 
  select(TRIMES, codcantf, CPERM) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de edificaciones" = sbtotal) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:15), ~replace(., is.na(.), 0)) %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  select(sort(names(.)))

# Completo
CC1 <- bind_rows(CC1_2022, CC1_2023 , CC1_2024, CC1_2025) %>% 
  mutate_all(~ifelse(is.na(.), 0, .))

rm(CC1_2022, CC1_2023 , CC1_2024, CC1_2025)

####### C2 Edificaciones a construir, por trimestres, según tipo de obra #######

# 2022
CC2_2022 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2022-I", 
                       "2022-II", 
                       "2022-III", 
                       "2022-IV")) %>% 
  select(TRIMES, CTIPOBR, CPERM) %>% 
  group_by(TRIMES, as_factor(CTIPOBR)) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de edificaciones" = sbtotal) %>% 
  pivot_wider(names_from = `as_factor(CTIPOBR)`, values_from = total) %>% 
  mutate_at(c(2:5), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CC2_2022)[5] <- "2022"

# 2023
CC2_2023 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2023-I",
                       "2023-II", 
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, CTIPOBR, CPERM) %>% 
  group_by(TRIMES, as_factor(CTIPOBR)) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de edificaciones" = sbtotal) %>% 
  pivot_wider(names_from = `as_factor(CTIPOBR)`, values_from = total) %>% 
  mutate_at(c(2:5), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup()%>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CC2_2023)[5] <- "2023"


# 2024
CC2_2024 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2024-I",
                       "2024-II",
                       "2024-III",
                       "2024-IV")) %>% 
  select(TRIMES, CTIPOBR, CPERM) %>% 
  group_by(TRIMES, as_factor(CTIPOBR)) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de edificaciones" = sbtotal) %>% 
  pivot_wider(names_from = `as_factor(CTIPOBR)`, values_from = total) %>% 
  mutate_at(c(2:5), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup()%>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CC2_2024)[5] <- "2024"

# 2025
CC2_2025 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2025-I", 
                       "2025-II",
                       "2025-III")) %>% 
  select(TRIMES, CTIPOBR, CPERM) %>% 
  group_by(TRIMES, as_factor(CTIPOBR)) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de edificaciones" = sbtotal) %>% 
  pivot_wider(names_from = `as_factor(CTIPOBR)`, values_from = total) %>% 
  mutate_at(c(2:5), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup()


# Completo
CC2 <- bind_rows(CC2_2022, CC2_2023 , CC2_2024, CC2_2025)

rm(CC2_2022, CC2_2023 , CC2_2024, CC2_2025)

####### C3 Edificaciones a construir, por trimestres, según uso de la edificación #######

# 2022
CC3_2022 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2022-I", 
                       "2022-II", 
                       "2022-III", 
                       "2022-IV")) %>% 
  select(TRIMES, CDISPUSX, CPERM) %>% 
  group_by(TRIMES, CDISPUSX) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>% 
  complete(CDISPUSX, fill = list(total = 0)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de edificaciones" = sbtotal) %>% 
  pivot_wider(names_from = CDISPUSX, values_from = total) %>% 
  mutate_at(c(2:17), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CC3_2022)[5] <- "2022"

# 2023
CC3_2023 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2023-I",
                       "2023-II", 
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, CDISPUSX, CPERM) %>% 
  group_by(TRIMES, CDISPUSX) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>%   
  complete(CDISPUSX, fill = list(total = 0)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de edificaciones" = sbtotal) %>% 
  pivot_wider(names_from = CDISPUSX, values_from = total) %>% 
  mutate_at(c(2:17), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CC3_2023)[5] <- "2023"


# 2024
CC3_2024 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2024-I", 
                       "2024-II",
                       "2024-III",
                       "2024-IV")) %>% 
  select(TRIMES, CDISPUSX, CPERM) %>% 
  group_by(TRIMES, CDISPUSX) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>%   
  complete(CDISPUSX, fill = list(total = 0)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de edificaciones" = sbtotal) %>% 
  pivot_wider(names_from = CDISPUSX, values_from = total) %>% 
  mutate_at(c(2:17), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CC3_2024)[5] <- "2024"


# 2025
CC3_2025 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2025-I", 
                       "2025-II",
                       "2025-III")) %>% 
  select(TRIMES, CDISPUSX, CPERM) %>% 
  group_by(TRIMES, CDISPUSX) %>% 
  summarise(total = sum(CPERM, na.rm = T)) %>%   
  complete(CDISPUSX, fill = list(total = 0)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de edificaciones" = sbtotal) %>% 
  pivot_wider(names_from = CDISPUSX, values_from = total) %>% 
  mutate_at(c(2:17), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup()

# Completo
CC3 <- bind_rows(CC3_2022, CC3_2023 , CC3_2024 , CC3_2025) %>% 
  select(- `No Aplica`) %>%
  mutate_at(c(1:16), ~replace(.,is.na(.),0))
names(CC3) <- c("Total de edificaciones",
                "Edificaciones con una vivienda",
                "Edificaciones con dos viviendas",
                "Edificaciones con tres o más viviendas",
                "Comercial",
                "Industrial",
                "Edificio administrativo (público)",
                "Educación particular",
                "Educación pública",
                "Cultura",
                "Complejos recreacionales",
                "Hospitales, clínicas y otros establecimientos de salud particular",
                "Hospitales, clínicas y otros establecimientos de salud pública",
                "Transporte y comunicaciones",
                "Mixto residencial y no residencial",
                "Otras")

rm(CC3_2022, CC3_2023 , CC3_2024 , CC3_2025)

####### C4 Edificaciones a construir, por trimestres, según acceso a internet – celular/convencional #######

### Cobertura internet/celular

# 2022
CC4_2022_1 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2022-I",
                       "2022-II",
                       "2022-III",
                       "2022-IV")) %>% 
  select(TRIMES, CPERM, CINTERN, careaur, CCEL) %>% 
  mutate(int_ur = case_when(careaur == 1 & CINTERN == 1  ~ 1,
                            careaur == 1 & CINTERN == 2  ~ 2,
                            T ~ 0)) %>%
  group_by(TRIMES, int_ur) %>% 
  summarise(total = sum(CPERM)) %>% 
  pivot_wider(names_from = int_ur, values_from = total) %>% 
  mutate_at(c(2:4), ~replace(., is.na(.), 0)) %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
names(CC4_2022_1) <- c("Ic urbana no aplica",
                       "Ic urbana Acceso",
                       "Ic urbana Sin acceso")

CC4_2022_2 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2022-I",
                       "2022-II",
                       "2022-III",
                       "2022-IV")) %>% 
  select(TRIMES, CPERM, CINTERN, careaur, CCEL) %>% 
  mutate(int_ru = case_when(careaur == 2 & CINTERN == 1  ~ 1,
                            careaur == 2 & CINTERN == 2  ~ 2,
                            T ~ 0)) %>%
  group_by(TRIMES, int_ru) %>% 
  summarise(total = sum(CPERM)) %>% 
  pivot_wider(names_from = int_ru, values_from = total) %>% 
  mutate_at(c(2:4), ~replace(., is.na(.), 0)) %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
names(CC4_2022_2) <- c("Ic rural no aplica",
                       "Ic rural Acceso",
                       "Ic rural Sin acceso")


CC4_2022 <- bind_cols(CC4_2022_1, CC4_2022_2) %>% 
  select(-"Ic urbana no aplica", -"Ic rural no aplica")
rownames(CC4_2022)[5] <- "2022"

rm(CC4_2022_1, CC4_2022_2)


# 2023
CC4_2023_1 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2023-I", 
                       "2023-II",
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, CPERM, CINTERN, careaur) %>% 
  mutate(int_ur = case_when(careaur == 1 & CINTERN == 1 ~ 1, 
                            careaur == 1 & CINTERN == 2 ~ 2,
                            T ~ 0)) %>%
  group_by(TRIMES, int_ur) %>% 
  summarise(total = sum(CPERM)) %>% 
  pivot_wider(names_from = int_ur, values_from = total) %>% 
  mutate_at(c(2:4), ~replace(., is.na(.), 0)) %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
names(CC4_2023_1) <- c("Ic urbana no aplica",
                       "Ic urbana Acceso",
                       "Ic urbana Sin acceso")

CC4_2023_2 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2023-I",
                       "2023-II", 
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, CPERM, CINTERN, careaur) %>% 
  mutate(int_ru = case_when(careaur == 2 & CINTERN == 1 ~ 1, 
                            careaur == 2 & CINTERN == 2 ~ 2,
                            T ~ 0)) %>%
  group_by(TRIMES, int_ru) %>% 
  summarise(total = sum(CPERM)) %>% 
  pivot_wider(names_from = int_ru, values_from = total) %>% 
  mutate_at(c(2:4), ~replace(., is.na(.), 0)) %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
names(CC4_2023_2) <- c("Ic rural no aplica",
                       "Ic rural Acceso",
                       "Ic rural Sin acceso")

CC4_2023 <- bind_cols(CC4_2023_1, CC4_2023_2) %>% 
  select(-"Ic urbana no aplica", - "Ic rural no aplica")
rownames(CC4_2023)[5] <- "2023"

rm(CC4_2023_1, CC4_2023_2)

## 2024

CC4_2024_1 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2024-I",
                       "2024-II",
                       "2024-III",
                       "2024-IV")) %>% 
  select(TRIMES, CPERM, CINTERN, careaur) %>% 
  mutate(int_ur = case_when(careaur == 1 & CINTERN == 1 ~ 1, 
                            careaur == 1 & CINTERN == 2 ~ 2,
                            T ~ 0)) %>%
  group_by(TRIMES, int_ur) %>% 
  summarise(total = sum(CPERM)) %>% 
  pivot_wider(names_from = int_ur, values_from = total) %>% 
  mutate_at(c(2:4), ~replace(., is.na(.), 0)) %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
names(CC4_2024_1) <- c("Ic urbana no aplica",
                       "Ic urbana Acceso",
                       "Ic urbana Sin acceso")

CC4_2024_2 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2024-I",
                       "2024-II",
                       "2024-III",
                       "2024-IV")) %>% 
  select(TRIMES, CPERM, CINTERN, careaur) %>% 
  mutate(int_ru = case_when(careaur == 2 & CINTERN == 1 ~ 1, 
                            careaur == 2 & CINTERN == 2 ~ 2,
                            T ~ 0)) %>%
  group_by(TRIMES, int_ru) %>% 
  summarise(total = sum(CPERM)) %>% 
  pivot_wider(names_from = int_ru, values_from = total) %>% 
  mutate_at(c(2:4), ~replace(., is.na(.), 0)) %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
names(CC4_2024_2) <- c("Ic rural no aplica",
                       "Ic rural Acceso",
                       "Ic rural Sin acceso")

CC4_2024 <- bind_cols(CC4_2024_1, CC4_2024_2) %>% 
  select(-"Ic urbana no aplica", - "Ic rural no aplica")

rownames(CC4_2024)[5] <- "2024"

rm(CC4_2024_1, CC4_2024_2)

# Completo 

CC4 <- bind_rows(CC4_2022, CC4_2023 ,CC4_2024 )

rm(CC4_2022, CC4_2023 , CC4_2024)

### Variables de acceso a internet nueva variable del 2025 

CC4_2025_1 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2025-I", 
                       "2025-II",
                       "2025-III")) %>% 
  select(TRIMES, CPERM, CINTERN, careaur) %>% 
  mutate(int_ur = case_when(careaur == 1 & CINTERN == 1 ~ 1, 
                            T ~ 0)) %>%
  group_by(TRIMES, int_ur) %>% 
  summarise(total = sum(CPERM)) %>% 
  pivot_wider(names_from = int_ur, values_from = total) %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  select(2)
names(CC4_2025_1) <- c("Ic urbana Acceso")


CC4_2025_2 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2025-I", 
                       "2025-II",
                       "2025-III")) %>% 
  select(TRIMES, CPERM, CINTERN, careaur) %>% 
  mutate(int_ur = case_when(careaur == 2 & CINTERN == 1 ~ 1, 
                            T ~ 0)) %>%
  group_by(TRIMES, int_ur) %>% 
  summarise(total = sum(CPERM)) %>% 
  pivot_wider(names_from = int_ur, values_from = total) %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  select(2)
names(CC4_2025_2) <- c("Ic rural Acceso")

### Variables de acceso a celular/convencional nueva variable del 2025 

CC4_2025_3 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2025-I", 
                       "2025-II",
                       "2025-III")) %>% 
  select(TRIMES, CPERM, ASTELF, careaur) %>% 
  mutate(int_ur = case_when(careaur == 1 & ASTELF == 1 ~ 1, 
                            T ~ 0)) %>%
  group_by(TRIMES, int_ur) %>% 
  summarise(total = sum(CPERM)) %>% 
  pivot_wider(names_from = int_ur, values_from = total) %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  select(2)
names(CC4_2025_3) <- c("Ic urbana Acceso 'Telf")


CC4_2025_4 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2025-I", 
                       "2025-II",
                       "2025-III")) %>% 
  select(TRIMES, CPERM, ASTELF, careaur) %>% 
  mutate(int_ur = case_when(careaur == 2 & ASTELF == 1 ~ 1, 
                            T ~ 0)) %>%
  group_by(TRIMES, int_ur) %>% 
  summarise(total = sum(CPERM)) %>% 
  pivot_wider(names_from = int_ur, values_from = total) %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  select(2)
names(CC4_2025_4) <- c("Ic rural Acceso - Telf")

# Unir variables

CC4_1 <- bind_cols(CC4_2025_1, CC4_2025_2, CC4_2025_3, CC4_2025_4) 

rm(CC4_2025_1, CC4_2025_2, CC4_2025_3, CC4_2025_4)

###### D Número de viviendas proyectadas ######*

####### D1 Viviendas a construir, por trimestres, según cantones #######

# 2022
CD1_2022 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2022-I", 
                       "2022-II", 
                       "2022-III", 
                       "2022-IV")) %>% 
  select(TRIMES, CNUVICAL, codcantf) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>%
  summarise(total = sum(CNUVICAL)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de viviendas" = sbtotal) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>%
  rename("Quito" = "Distrito Metropolitano De Quito") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CD1_2022)[5] <- "2022"

# 2023
CD1_2023 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2023-I", 
                       "2023-II",
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, CNUVICAL, codcantf) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>%
  summarise(total = sum(CNUVICAL)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de viviendas" = sbtotal) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>%
  rename("Quito" = "Distrito Metropolitano De Quito") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CD1_2023)[5] <- "2023"

# 2024
CD1_2024 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2024-I",
                       "2024-II",
                       "2024-III",
                       "2024-IV")) %>% 
  select(TRIMES, CNUVICAL, codcantf) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>%
  summarise(total = sum(CNUVICAL)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de viviendas" = sbtotal) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>%
  rename("Quito" = "Distrito Metropolitano De Quito") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CD1_2024)[5] <- "2024"


# 2025
CD1_2025 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2025-I", 
                       "2025-II",
                       "2025-III")) %>% 
  select(TRIMES, CNUVICAL, codcantf) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>%
  summarise(total = sum(CNUVICAL)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de viviendas" = sbtotal) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:15), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  select(sort(names(.)))


# Completo
CD1 <- bind_rows(CD1_2022, CD1_2023, CD1_2024, CD1_2025) %>% 
  mutate_all(~ifelse(is.na(.), 0, .))

rm( CD1_2022, CD1_2023, CD1_2024, CD1_2025)

####### D2 Viviendas a construir, por trimestres, según metros cuadrados #######

# 2022
CD2_2022 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2022-I", 
                       "2022-II", 
                       "2022-III", 
                       "2022-IV")) %>% 
  select(TRIMES, CNUVICAL, CDISPARC) %>% 
  group_by(TRIMES, as_factor(CDISPARC)) %>%
  summarise(total = sum(CNUVICAL)) %>%
  mutate(subtotal = sum(total)) %>% 
  pivot_wider(names_from = `as_factor(CDISPARC)`, values_from = total) %>% 
  mutate_at(c(2:7), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CD2_2022)[5] <- "2022"

# 2023
CD2_2023 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2023-I" , 
                       "2023-II", 
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, CNUVICAL, CDISPARC) %>% 
  group_by(TRIMES, as_factor(CDISPARC)) %>%
  summarise(total = sum(CNUVICAL)) %>%
  mutate(subtotal = sum(total)) %>% 
  pivot_wider(names_from = `as_factor(CDISPARC)`, values_from = total) %>% 
  mutate_at(c(2:7), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CD2_2023)[5] <- "2023"

# 2024
CD2_2024 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2024-I",
                       "2024-II",
                       "2024-III",
                       "2024-IV")) %>% 
  select(TRIMES, CNUVICAL, CDISPARC) %>% 
  group_by(TRIMES, as_factor(CDISPARC)) %>%
  summarise(total = sum(CNUVICAL)) %>%
  mutate(subtotal = sum(total)) %>% 
  pivot_wider(names_from = `as_factor(CDISPARC)`, values_from = total) %>% 
  mutate_at(c(2:7), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CD2_2024)[5] <- "2024"


# 2025
CD2_2025 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2025-I", 
                       "2025-II",
                       "2025-III")) %>% 
  select(TRIMES, CNUVICAL, CDISPARC) %>% 
  group_by(TRIMES, as_factor(CDISPARC)) %>%
  summarise(total = sum(CNUVICAL)) %>%
  mutate(subtotal = sum(total)) %>% 
  pivot_wider(names_from = `as_factor(CDISPARC)`, values_from = total) %>% 
  mutate_at(c(2:7), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup()


# Completo
CD2 <- bind_rows(CD2_2022, CD2_2023, CD2_2024, CD2_2025)
names(CD2) <- c("Total de viviendas",
                "Menos de 100 m2",
                "100 a 199 m2",
                "200 a 299 m2",
                "300 a 399 m2",
                "400 a 499 m2",
                "500 y más m2")

rm(CD2_2022, CD2_2023, CD2_2024, CD2_2025)

####### D3 Viviendas a construir, por trimestres, según número de dormitorios #######

# 2022
CD3_2022 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2022-I", 
                       "2022-II", 
                       "2022-III", 
                       "2022-IV")) %>% 
  select(TRIMES, CNUVICAL, NUDOCAL) %>% 
  mutate(NUDOCAL1 = case_when(NUDOCAL == 0  ~ 0,
                              NUDOCAL >= 1 & NUDOCAL <= 1.49 ~ 1,
                              NUDOCAL >= 1.5 & NUDOCAL <= 2.49 ~ 2,
                              NUDOCAL >= 2.5 & NUDOCAL <= 3.49 ~ 3,
                              NUDOCAL >= 3.5 & NUDOCAL <= 4.49 ~ 4,
                              NUDOCAL >= 4.5 ~ 5,
                              T ~ 0)) %>%
  group_by(TRIMES, as_factor(NUDOCAL1)) %>% 
  summarise(total = sum(CNUVICAL)) %>% 
  pivot_wider(names_from = `as_factor(NUDOCAL1)`, values_from = total) %>% 
  mutate_at(c(2:7), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  rename("5 y más" = "5")
rownames(CD3_2022)[5] <- "2022"

# 2023
CD3_2023 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2023-I",
                       "2023-II", 
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, CNUVICAL, NUDOCAL) %>% 
  mutate(NUDOCAL1 = case_when(NUDOCAL == 0  ~ 0,
                              NUDOCAL == 1  ~ 1,
                              NUDOCAL == 2  ~ 2,
                              NUDOCAL == 3  ~ 3,
                              NUDOCAL == 4  ~ 4,
                              NUDOCAL >= 5 ~ 5,
                              T ~ 0)) %>%
  group_by(TRIMES, as_factor(NUDOCAL1)) %>% 
  summarise(total = sum(CNUVICAL)) %>% 
  pivot_wider(names_from = `as_factor(NUDOCAL1)`, values_from = total) %>% 
  mutate_at(c(2:7), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  rename("5 y más" = "5")
rownames(CD3_2023)[5] <- "2023"

# 2024
CD3_2024 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2024-I",
                       "2024-II",
                       "2024-III",
                       "2024-IV")) %>% 
  select(TRIMES, CNUVICAL, NUDOCAL) %>% 
  mutate(NUDOCAL1 = case_when(NUDOCAL == 0  ~ 0,
                              NUDOCAL == 1  ~ 1,
                              NUDOCAL == 2  ~ 2,
                              NUDOCAL == 3  ~ 3,
                              NUDOCAL == 4  ~ 4,
                              NUDOCAL >= 5 ~ 5,
                              T ~ 0)) %>%
  group_by(TRIMES, as_factor(NUDOCAL1)) %>% 
  summarise(total = sum(CNUVICAL)) %>% 
  pivot_wider(names_from = `as_factor(NUDOCAL1)`, values_from = total) %>% 
  mutate_at(c(2:7), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  rename("5 y más" = "5")
rownames(CD3_2024)[5] <- "2024"


# 2025
CD3_2025 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2025-I", 
                       "2025-II",
                       "2025-III")) %>% 
  select(TRIMES, CNUVICAL, NUDOCAL) %>% 
  mutate(NUDOCAL1 = case_when(NUDOCAL == 0  ~ 0,
                              NUDOCAL == 1  ~ 1,
                              NUDOCAL == 2  ~ 2,
                              NUDOCAL == 3  ~ 3,
                              NUDOCAL == 4  ~ 4,
                              NUDOCAL >= 5 ~ 5,
                              T ~ 0)) %>%
  group_by(TRIMES, as_factor(NUDOCAL1)) %>% 
  summarise(total = sum(CNUVICAL)) %>% 
  pivot_wider(names_from = `as_factor(NUDOCAL1)`, values_from = total) %>% 
  mutate_at(c(2:7), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  rename("5 y más" = "5")


# Completo
CD3 <- bind_rows(CD3_2022, CD3_2023, CD3_2024, CD3_2025) %>% 
  select(-"0")

rm(CD3_2022, CD3_2023, CD3_2024, CD3_2025)

###### E Superficie del terreno y áreas a construir ######* 

####### E1 Superficie del terreno y área a construir, por trimestres, según uso de la edificación #######

# 2022
CE1_2022 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2022-I", 
                       "2022-II", 
                       "2022-III", 
                       "2022-IV")) %>% 
  select(TRIMES, CSUTE, CARCO, CARES, CARNRES, CARDPAR, CAESV) %>% 
  group_by(TRIMES) %>% 
  summarise(stt = sum(CSUTE),
            atc = sum(CARCO),
            res = sum(CARES),
            nres = sum(CARNRES),
            par = sum(CARDPAR),
            aev = sum(CAESV)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CE1_2022)[5] <- "2022"

# 2023
CE1_2023 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2023-I" ,
                       "2023-II", 
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, CSUTE, CARCO, CARES, CARNRES, CARDPAR, CAESV) %>% 
  group_by(TRIMES) %>% 
  summarise(stt = sum(CSUTE),
            atc = sum(CARCO),
            res = sum(CARES),
            nres = sum(CARNRES),
            par = sum(CARDPAR),
            aev = sum(CAESV)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CE1_2023)[5] <- "2023"


# 2024
CE1_2024 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2024-I",
                       "2024-II",
                       "2024-III",
                       "2024-IV")) %>% 
  select(TRIMES, CSUTE, CARCO, CARES, CARNRES, CARDPAR, CAESV) %>% 
  group_by(TRIMES) %>% 
  summarise(stt = sum(CSUTE),
            atc = sum(CARCO),
            res = sum(CARES),
            nres = sum(CARNRES),
            par = sum(CARDPAR),
            aev = sum(CAESV)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CE1_2024)[5] <- "2024"

# 2025
CE1_2025 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2025-I", 
                       "2025-II",
                       "2025-III")) %>% 
  select(TRIMES, CSUTE, CARCO, CARES, CARNRES, CARDPAR, CAESV) %>% 
  group_by(TRIMES) %>% 
  summarise(stt = sum(CSUTE),
            atc = sum(CARCO),
            res = sum(CARES),
            nres = sum(CARNRES),
            par = sum(CARDPAR),
            aev = sum(CAESV)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup()


# Completo

CE1 <- bind_rows(CE1_2022, CE1_2023 , CE1_2024, CE1_2025) %>% 
  round(., digits = 0)
names(CE1) <- c("Superficie total del terreno",
                "Total",
                "Residencial",
                "No residencial",
                "Parqueadero*",
                "Área para espacios verdes")

rm(CE1_2022, CE1_2023 , CE1_2024, CE1_2025)


###### F Valor estimado de la edificación ######*

####### F1 Valor estimado del financiamiento de la edificación, por trimestres, según tipo de obra #######

# 2022
CF1_2022 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2022-I", 
                       "2022-II", 
                       "2022-III", 
                       "2022-IV")) %>% 
  select(TRIMES, CVAE, CTIPOBR) %>% 
  group_by(TRIMES, as_factor(CTIPOBR)) %>% 
  summarise(total = sum(CVAE)) %>% 
  mutate("Monto total de financiamiento" = sum(total)) %>% 
  pivot_wider(names_from = `as_factor(CTIPOBR)`, values_from = total) %>% 
  mutate_at(c(2:5), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CF1_2022)[5] <- "2022"

# 2023
CF1_2023 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2023-I" , 
                       "2023-II",
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, CVAE, CTIPOBR) %>% 
  group_by(TRIMES, as_factor(CTIPOBR)) %>% 
  summarise(total = sum(CVAE)) %>% 
  mutate("Monto total de financiamiento" = sum(total)) %>% 
  pivot_wider(names_from = `as_factor(CTIPOBR)`, values_from = total) %>% 
  mutate_at(c(2:5), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CF1_2023)[5] <- "2023"

# 2024
CF1_2024 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2024-I",
                       "2024-II",
                       "2024-III",
                       "2024-IV")) %>% 
  select(TRIMES, CVAE, CTIPOBR) %>% 
  group_by(TRIMES, as_factor(CTIPOBR)) %>% 
  summarise(total = sum(CVAE)) %>% 
  mutate("Monto total de financiamiento" = sum(total)) %>% 
  pivot_wider(names_from = `as_factor(CTIPOBR)`, values_from = total) %>% 
  mutate_at(c(2:5), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CF1_2024)[5] <- "2024"


# 2025
CF1_2025 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2025-I",
                       "2025-II",
                       "2025-III")) %>% 
  select(TRIMES, CVAE, CTIPOBR) %>% 
  group_by(TRIMES, as_factor(CTIPOBR)) %>% 
  summarise(total = sum(CVAE)) %>% 
  mutate("Monto total de financiamiento" = sum(total)) %>% 
  pivot_wider(names_from = `as_factor(CTIPOBR)`, values_from = total) %>% 
  mutate_at(c(2:5), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup()


# Completo
CF1 <- bind_rows(CF1_2022, CF1_2023 ,CF1_2024, CF1_2025 )

rm( CF1_2022, CF1_2023 ,CF1_2024, CF1_2025 )


####### F2 Valor estimado del financiamiento de la edificación, por trimestres, según uso de la edificación #######

# 2022
CF2_2022 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2022-I", 
                       "2022-II", 
                       "2022-III", 
                       "2022-IV")) %>% 
  select(TRIMES, CVAE, CDISPUSX) %>% 
  group_by(TRIMES, CDISPUSX) %>% 
  summarise(total = sum(CVAE)) %>% 
  complete(CDISPUSX,  fill = list(total = 0)) %>%
  mutate("Monto total de financiamiento" = sum(total)) %>% 
  pivot_wider(names_from = CDISPUSX, values_from = total) %>% 
  mutate_at(c(2:18), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CF2_2022)[5] <- "2022"

# 2023
CF2_2023 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2023-I",
                       "2023-II",
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, CVAE, CDISPUSX) %>% 
  group_by(TRIMES, CDISPUSX) %>% 
  summarise(total = sum(CVAE)) %>% 
  complete(CDISPUSX,  fill = list(total = 0)) %>%
  mutate("Monto total de financiamiento" = sum(total)) %>% 
  pivot_wider(names_from = CDISPUSX, values_from = total) %>% 
  mutate_at(c(2:18), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CF2_2023)[5] <- "2023"


# 2024
CF2_2024 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2024-I",
                       "2024-II",
                       "2024-III",
                       "2024-IV")) %>% 
  select(TRIMES, CVAE, CDISPUSX) %>% 
  group_by(TRIMES, CDISPUSX) %>% 
  summarise(total = sum(CVAE)) %>% 
  complete(CDISPUSX,  fill = list(total = 0)) %>%
  mutate("Monto total de financiamiento" = sum(total)) %>% 
  pivot_wider(names_from = CDISPUSX, values_from = total) %>% 
  mutate_at(c(2:18), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum)))
rownames(CF2_2024)[5] <- "2024"


# 2025
CF2_2025 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2025-I", 
                       "2025-II",
                       "2025-III")) %>% 
  select(TRIMES, CVAE, CDISPUSX) %>% 
  group_by(TRIMES, CDISPUSX) %>% 
  summarise(total = sum(CVAE)) %>% 
  complete(CDISPUSX,  fill = list(total = 0)) %>%
  mutate("Monto total de financiamiento" = sum(total)) %>% 
  pivot_wider(names_from = CDISPUSX, values_from = total) %>% 
  mutate_at(c(2:18), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>%
  ungroup()

# Completo
CF2 <- bind_rows(CF2_2022, CF2_2023 , CF2_2024, CF2_2025) %>% 
  select(- `No Aplica`) %>%
  mutate_at(c(1:16), ~replace(.,is.na(.),0))
names(CF2) <- c("Monto total de financiamiento",
                "Edificaciones con una vivienda",
                "Edificaciones con dos viviendas",
                "Edificaciones con tres o más viviendas",
                "Comercial",
                "Industrial",
                "Edificio administrativo (público)",
                "Educación particular",
                "Educación pública",
                "Cultura",
                "Complejos recreacionales",
                "Hospitales, clínicas y otros establecimientos de salud particular",
                "Hospitales, clínicas y otros establecimientos de salud pública",
                "Transporte y comunicaciones",
                "Mixto residencial y no residencial",
                "Otras")

rm(CF2_2022, CF2_2023 , CF2_2024, CF2_2025)


####### F3 Valor estimado de la edificación, por trimestres, según cantones #######


# 2022
CF3_2022 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2022-I", 
                       "2022-II", 
                       "2022-III", 
                       "2022-IV")) %>% 
  select(TRIMES, CVAE, codcantf) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(CVAE)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de financiamiento" = sbtotal) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>%
  rename("Quito" = "Distrito Metropolitano De Quito") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  ungroup() %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CF3_2022)[5] <- "2022"

# 2023
CF3_2023 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2023-I",
                       "2023-II",
                       "2023-III",
                       "2023-IV")) %>% 
  select(TRIMES, CVAE, codcantf) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(CVAE)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de financiamiento" = sbtotal) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:16), ~replace(., is.na(.), 0)) %>%
  rename("Quito" = "Distrito Metropolitano De Quito") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CF3_2023)[5] <- "2023"


# 2024
CF3_2024 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2024-I",
                       "2024-II",
                       "2024-III",
                       "2024-IV")) %>% 
  select(TRIMES, CVAE, codcantf) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(CVAE)) %>% 
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de financiamiento" = sbtotal) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:15), ~replace(., is.na(.), 0)) %>%
  rename("Quito" = "Distrito Metropolitano De Quito") %>% 
  column_to_rownames(var = "TRIMES") %>% 
  bind_rows(summarise(., across(where(is.numeric), sum))) %>% 
  select(sort(names(.)))
rownames(CF3_2024)[5] <- "2024"


# 2025
CF3_2025 <- ESED_2025_III %>% 
  filter(TRIMES %in% c("2025-I", 
                       "2025-II",
                       "2025-III")) %>% 
  select(TRIMES, CVAE, codcantf) %>% 
  group_by(TRIMES, as_factor(codcantf)) %>% 
  summarise(total = sum(CVAE)) %>%
  mutate(sbtotal = sum(total, na.rm = T)) %>%
  rename("Total de financiamiento" = sbtotal) %>% 
  pivot_wider(names_from = `as_factor(codcantf)`, values_from = total) %>% 
  mutate_at(c(2:15), ~replace(., is.na(.), 0)) %>%
  column_to_rownames(var = "TRIMES") %>% 
  select(sort(names(.)))


# Completo
CF3 <- bind_rows(CF3_2022, CF3_2023 , CF3_2024, CF3_2025) %>% 
  mutate_all(~ifelse(is.na(.), 0, .))

rm(CF3_2022, CF3_2023 , CF3_2024, CF3_2025)

# ============================================================================ #
#####               DISEÑO Y EXPORTACIÓN DE TABULADOS A EXCEL              ##### 
# ============================================================================ #

######                Definición de hojas de excel                         #####*

tab_ESED <- createWorkbook()
addWorksheet(tab_ESED, "ÍNDICE", zoom = 70)
addWorksheet(tab_ESED, "A1", zoom = 70)
addWorksheet(tab_ESED, "B1", zoom = 70)
addWorksheet(tab_ESED, "C1", zoom = 70)
addWorksheet(tab_ESED, "C2", zoom = 70)
addWorksheet(tab_ESED, "C3", zoom = 70)
addWorksheet(tab_ESED, "C4", zoom = 70)
addWorksheet(tab_ESED, "D1", zoom = 70)
addWorksheet(tab_ESED, "D2", zoom = 70)
addWorksheet(tab_ESED, "D3", zoom = 70)
addWorksheet(tab_ESED, "E1", zoom = 70)
addWorksheet(tab_ESED, "F1", zoom = 70)
addWorksheet(tab_ESED, "F2", zoom = 70)
addWorksheet(tab_ESED, "F3", zoom = 70)

# ============================================================================ #
#########                     Configuración General del Índice          ########
# ============================================================================ #

# Creación del índice

c1 <- c("A", "A1",
        "B", "B1",
        "C", "C1", "C2", "C3", "C4",
        "D", "D1", "D2", "D3",
        "E", "E1",
        "F", "F1", "F2", "F3")

c2 <- c("Cuadro resumen", # A
        "Resultados de las principales variables",
        "Número de permisos de construcción", # B
        "Permisos de construcción, por trimestres, según cantones",
        "Número de edificaciones a construir", # C
        "Edificaciones a construir, por trimestres, según cantones",
        "Edificaciones a construir, por trimestres, según tipo de obra",
        "Edificaciones a construir, por trimestres, según uso de la edificación",
        "Edificaciones a construir, por trimestres, según acceso a internet – telefonía celular/convencional",
        "Número de viviendas proyectadas", # D
        "Viviendas a construir, por trimestres, según cantones",
        "Viviendas a construir, por trimestres, según metros cuadrados",
        "Viviendas a construir, por trimestres, según número de dormitorios",
        "Superficie del terreno y áreas a construir", # E
        "Superficie del terreno y área a construir, por trimestres, según uso de la edificación",
        "Valor estimado de las edificaciones", # F
        "Valor estimado del financiamiento de las edificaciones, por trimestres, según tipo de obra",
        "Valor estimado del financiamiento de las edificaciones, por trimestres, según uso de la edificación",
        "Valor estimado de las edificaciones, por trimestres, según cantones")

ind <- data.frame(c1, c2)		
colnames(ind) <- c("Cuadro No.", "Contenido")

# Diseño y estilos el indice

"Primera fila: Cuadro de Títulos (Cuadro No. - Contenido)"
titenc <- createStyle(fontName = "Century Gothic", 
                      fontSize = 13,
                      textDecoration = "bold",
                      fontColour = "#FFFFFF", 
                      fgFill = "#043F5C",
                      borderColour = "#FFFFFF",
                      halign = "center",
                      valign = "center",
                      border = "TopBottomLeftRight",
                      wrapText = T)

"Filas A-B..."
fill_abcd <- createStyle(fontName = "Century Gothic", 
                         fontSize = 13,
                         textDecoration = "bold",
                         fontColour = "#595959", 
                         fgFill = "#87D3D5",
                         borderColour = "#595959",
                         halign = "left",
                         valign = "center",
                         border = "TopBottomLeftRight",
                         wrapText = T)

"Centrado filas A-B..."
fill_rest1 <- createStyle(fontName = "Century Gothic", 
                         fontSize = 13,
                         textDecoration = "bold",
                         fontColour = "#595959", 
                         fgFill = "#87D3D5",
                         borderColour = "#595959",
                         halign = "center",
                         valign = "center",
                         border = "TopBottomLeftRight",
                         wrapText = T)

"Filas tipo A1-B1..."
fill_a1 <- createStyle(fontName = "Century Gothic",
                       fontSize = 13,
                       fontColour = "#595959", 
                       borderColour = "#595959",
                       halign = "left",
                       valign = "center",
                       border = "TopBottomLeftRight",
                       wrapText = T)

"Centrado filas A1-B1..."
fill_rest2 <- createStyle(fontName = "Century Gothic", 
                          fontSize = 13,
                          fontColour = "#595959", 
                          borderColour = "#595959",
                          halign = "center",
                          valign = "center",
                          border = "TopBottomLeftRight",
                          wrapText = TRUE) 

"Footnotes"
foot1 <- createStyle(fontName = "Century Gothic", 
                    fontSize = 11,
                    fontColour = "#595959",
                    textDecoration = "bold",
                    halign = "left",
                    valign = "center")

foot2 <- createStyle(fontName = "Century Gothic", 
                    fontSize = 11,
                    fontColour = "#595959",
                    halign = "left",
                    valign = "center")

# Configuración 

"Imagen"
insertImage(tab_ESED,
            "ÍNDICE",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\ing_i.png", # cambiar
            height = 3.61,
            width = 38,00,
            startCol = 1,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "ÍNDICE", 
              rows = 1, 
              heights = "100") 

setRowHeights(tab_ESED, 
              "ÍNDICE", 
              rows = 4:41, 
              heights = "26.25")

"Encabezado"
addStyle(tab_ESED,
         "ÍNDICE",
         titenc,
         cols = 2:3,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "ÍNDICE", 
             cols = 2, 
             widths = "20")

setColWidths(tab_ESED,
             "ÍNDICE", 
             cols = 1, 
             widths = "8")

setColWidths(tab_ESED,
             "ÍNDICE", 
             cols = 3, 
             widths = "170")

"Datos"
writeData(tab_ESED,
          "ÍNDICE",
          ind, 
          startRow = 4, 
          startCol = 2,  
          colNames = T, 
          withFilter = F)

"Diseño filas A-B..."
addStyle(tab_ESED, 
         "ÍNDICE", 
         fill_abcd, 
         cols = 2:3, 
         rows = c(5, 7, 9, 14, 18, 20), 
         gridExpand = T)

"Diseño columna A-B..."
addStyle(tab_ESED,
         "ÍNDICE",
         fill_rest1,
         cols = 2,
         rows = c(5, 7, 9, 14, 18, 20),
         gridExpand = T)

"Diseño filas A1-B1..."
addStyle(tab_ESED,
         "ÍNDICE",
         fill_a1,
         cols = 2:3,
         rows = c(6, 8, 10:13, 15:17, 19, 21:23),
         gridExpand = T)

"Diseño columna A1-B1..."
addStyle(tab_ESED,
         "ÍNDICE",
         fill_rest2,
         cols = 2,
         rows = c(6, 8, 10:13, 15:17, 19, 21:23),
         gridExpand = T)

"Diseño pie de página"
setRowHeights(tab_ESED, 
              "ÍNDICE", 
              rows = 26:27, 
              heights = "12")

setRowHeights(tab_ESED, 
              "ÍNDICE", 
              rows = 26:32, 
              heights = "18")

addStyle(tab_ESED,
         "ÍNDICE",
         foot1,
         cols = 2,
         rows = 26:32,
         gridExpand = T)

addStyle(tab_ESED,
         "ÍNDICE",
         foot2,
         cols = 3,
         rows = 26:32,
         gridExpand = T)
         
writeData(tab_ESED, 
          "ÍNDICE", 
          c("Dirección responsable de la información estadística y contenidos:",
            "Dirección de Estadísticas Económicas"),
          startCol = 2, 
          startRow = 26) 

writeData(tab_ESED, 
          "ÍNDICE", 
          c("Realizadores",
            "Elaboración:",
            "Revisión:",
            "Aprobación:"),
          startCol = 2, 
          startRow = 29) 

writeData(tab_ESED, 
          "ÍNDICE", 
          c("Valery Paz y Miño / Enrique Vallejo",
            "Lorena Ramos / Roberto Chaves",
            "Diana Barco"),
          startCol = 3, 
          startRow = 30) 


# ============================================================================ #
#####################   Configuración  General de los Tabulados   #############
# ============================================================================ #

"Título del cuadro"
tit_c <- createStyle(fontName = "Century Gothic", 
                     fontSize = 14,
                     textDecoration = "bold",
                     fontColour = "#595959",
                     halign = "left",
                     valign = "center",
                     wrapText = T)

"Formato año suma"
for_suma1 <- createStyle(fontName = "Century Gothic", 
                         fontSize = 13,
                         textDecoration = "bold",
                         fontColour = "#595959", 
                         fgFill = "#87D3D5",
                         borderColour = "#595959",
                         halign = "center",
                         valign = "center",
                         border = "TopBottomLeftRight",
                         wrapText = T)

"Formato sumas de datos"
for_suma2 <- createStyle(fontName = "Century Gothic", 
                         fontSize = 13,
                         textDecoration = "bold",
                         fontColour = "#595959", 
                         fgFill = "#87D3D5",
                         borderColour = "#595959",
                         numFmt = "#,##0",
                         halign = "center",
                         valign = "center",
                         border = "TopBottomLeftRight",
                         wrapText = T)

"Formato trimestres datos"
for_dat1 <- createStyle(fontName = "Century Gothic",
                        fontSize = 13,
                        fontColour = "#595959",
                        borderColour = "#595959",
                        halign = "center",
                        valign = "center",
                        border = "TopBottomLeftRight",
                        wrapText = T)

"Formato datos"
for_dat2 <- createStyle(fontName = "Century Gothic",
                        fontSize = 13,
                        fontColour = "#595959",
                        borderColour = "#595959",
                        numFmt = "#,##0",
                        halign = "center",
                        valign = "center",
                        border = "TopBottomLeftRight",
                        wrapText = T)

"Fuente"
fuente <- createStyle(fontName = "Century Gothic", 
                      fontSize = 11,
                      fontColour = "#595959",
                      textDecoration = "bold",
                      halign = "right",
                      valign = "center",
                      wrapText = F)

"Fuente 2 y Nota"
fuente2 <- createStyle(fontName = "Century Gothic",
                       fontSize = 11,
                       fontColour = "#595959",
                       halign = "left",
                       valign = "center",
                       wrapText = F)

# ============================================================================ #
#######################   Configuración Tabulado A1   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "A1",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\ing_ca1.png",
            height = 3.80,
            width = 43,22,
            startCol = 1,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED,
              "A1", 
              rows = 1, 
              heights = "100") 

"Título"
writeData(tab_ESED,
          "A1",
          "Cuadro A1",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "A1",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

setRowHeights(tab_ESED,
              "A1",
              rows = c(4),
              heights = "60")

"Encabezado"
addStyle(tab_ESED,
         "A1",
         titenc,
         cols = 2:8,
         rows = 4,
         gridExpand = T)

"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "A1", 
             cols = 2:8, 
             widths = "27")

setColWidths(tab_ESED,
             "A1", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "A1",
              rows = 5:22, # cambiar para exportar
              heights = "40") 

"Datos y Diseño"
writeData(tab_ESED,
          "A1",
          CA1,
          startCol = 2,
          startRow = 4,
          colNames = T,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "A1",
          "Trimestres",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "A1",
         for_dat1,
         cols = 2,
         rows = c(5:8, 10:13, 15:18, 20:22), # cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "A1",
         for_dat2,
         cols = 3:8,
         rows = c(5:8, 10:13, 15:18, 20:22), # cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "A1",
         for_suma1,
         cols = 2,
         rows = c(9, 14, 19), 
         gridExpand = T)

addStyle(tab_ESED,
         "A1",
         for_suma2,
         cols = 3:8,
         rows = c(9, 14, 19 ),
         gridExpand = T)

"Fuentes y notas"

setRowHeights(tab_ESED, 
              "A1", 
              rows = 23:30, # cambiar para exportar
              heights = "18")

writeData(tab_ESED,
          "A1",
          "Elaboración: Instituto Nacional de Estadística y Censos (INEC).",
          startCol = 2,
          startRow = 24) # cambiar para exportar

writeData(tab_ESED,
          "A1",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2022 - 2025.",
          startCol = 2,
          startRow = 25) # cambiar para exportar

writeData(tab_ESED,
          "A1",
          "Nota 1: Los resultados del año 2022 y 2023  son cifras semi-definitivas, del año 2024 son cifras provisionales, y del año 2025 son cifras preliminares.",
          startCol = 2,
          startRow = 26) # cambiar para exportar

writeData(tab_ESED,
          "A1",
          "Nota 2: Para el I, II y III trimestre 2025, no se dispone de la información del D. M. Quito. La información será actualizada, una vez que se disponga de los datos.",
          startCol = 2,
          startRow = 27) # cambiar para exportar

addStyle(tab_ESED,
         "A1",
         fuente2,
         cols = 2,
         rows = 23:30, # cambiar para exportar
         gridExpand = T)

# ============================================================================ #
#######################   Configuración Tabulado B1   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "B1",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\ing_cb1.png",
            height = 4.62,
            width = 70,20,
            startCol = 1,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "B1", 
              rows = 1, 
              heights = "130") 

"Título"
writeData(tab_ESED,
          "B1",
          "Cuadro B1",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "B1",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

setRowHeights(tab_ESED,
              "B1",
              rows = c(4),
              heights = "60")

"Encabezado"
addStyle(tab_ESED,
         "B1",
         titenc,
         cols = 2:18,
         rows = 4,
         gridExpand = T)


"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "B1", 
             cols = 2, 
             widths = "35")

setColWidths(tab_ESED,
             "B1", 
             cols = 3:18, 
             widths = "20")

setColWidths(tab_ESED,
             "B1", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "B1",
              rows = 5:22,  # cambiar para exportar
              heights = "40")


"Datos y Diseño"
writeData(tab_ESED,
          "B1",
          CB1,
          startCol = 2,
          startRow = 4,
          colNames = T,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "B1",
          "Trimestres / Cantones",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "B1",
         for_dat1,
         cols = 2,
         rows = c(5:8, 10:13, 15:18, 20:22), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "B1",
         for_dat2,
         cols = 3:18,
         rows = c(5:8, 10:13, 15:18, 20:22), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "B1",
         for_suma1,
         cols = 2,
         rows = c(9, 14, 19 ),
         gridExpand = T)

addStyle(tab_ESED,
         "B1",
         for_suma2,
         cols = 3:18,
         rows = c(9, 14, 19 ),
         gridExpand = T)

"Fuentes y notas"
setRowHeights(tab_ESED, 
              "B1", 
              rows = 23:30,  # cambiar para exportar
              heights = "18")

writeData(tab_ESED,
          "B1",
          "Elaboración: Instituto Nacional de Estadística y Censos (INEC).",
          startCol = 2,
          startRow = 24) # cambiar para exportar

writeData(tab_ESED,
          "B1",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2022 - 2025.",
          startCol = 2,
          startRow = 25) # cambiar para exportar


writeData(tab_ESED,
          "B1",
          "Nota 1: Los resultados del año 2022 y 2023  son cifras semi-definitivas, del año 2024 son cifras provisionales, y del año 2025 son cifras preliminares.",
          startCol = 2,
          startRow = 26) # cambiar

writeData(tab_ESED,
          "B1",
          "Nota 2: Para el I, II y III trimestre 2025, no se dispone de la información del D. M. Quito. La información será actualizada, una vez que se disponga de los datos.",
          startCol = 2,
          startRow = 27) # cambiar

addStyle(tab_ESED,
         "B1",
         fuente2,
         cols = 2,
         rows = 23:30, # cambiar
         gridExpand = T)


# Auxiliar para valores con cero (0)

# Definir las combinaciones de columnas y filas en la lista

posiciones <- list(
  c(14, 20), c(14, 21), c(14,22) # Columna 14
)

for (pos in posiciones) {
  writeData(tab_ESED,
            "B1",
            "-",
            startCol = pos[1],  # Columna
            startRow = pos[2],  # Fila
            colNames = FALSE)
}

addStyle(tab_ESED,
         "B1",
         for_dat2,
         cols = 14,
         rows = c(20:22), #cambiar para exportar
         gridExpand = T)


# ============================================================================ #
#######################   Configuración Tabulado C1   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "C1",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\ing_cc1.png",
            height = 4.60,
            width = 70.20,
            startCol = 1,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "C1", 
              rows = 1, 
              heights = "130") 

"Título"
writeData(tab_ESED,
          "C1",
          "Cuadro C1",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "C1",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

setRowHeights(tab_ESED,
              "C1",
              rows = c(4),
              heights = "60")

"Encabezado"
addStyle(tab_ESED,
         "C1",
         titenc,
         cols = 2:18,
         rows = 4,
         gridExpand = T)

"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "C1", 
             cols = 2, 
             widths = "35")

setColWidths(tab_ESED,
             "C1", 
             cols = 3:18, 
             widths = "20")

setColWidths(tab_ESED,
             "C1", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "C1",
              rows = 5:22,
              heights = "40") #  cambiar para exportar

"Datos y Diseño"
writeData(tab_ESED,
          "C1",
          CC1,
          startCol = 2,
          startRow = 4,
          colNames = T,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "C1",
          "Trimestres / Cantones",
          startCol = 2,
          startRow = 4)


addStyle(tab_ESED,
         "C1",
         for_dat1,
         cols = 2,
         rows = c(5:8, 10:13, 15:18, 20:22), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C1",
         for_dat2,
         cols = 3:18,
         rows = c(5:8, 10:13, 15:18, 20:22), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C1",
         for_suma1,
         cols = 2,
         rows = c(9, 14, 19),
         gridExpand = T)

addStyle(tab_ESED,
         "C1",
         for_suma2,
         cols = 3:18,
         rows = c(9, 14, 19  ),
         gridExpand = T)


"Fuentes y notas"
setRowHeights(tab_ESED, 
             "C1", 
             rows = 23:30, # cambiar para exportar
             heights = "18")

writeData(tab_ESED,
          "C1",
          "Elaboración: Instituto Nacional de Estadística y Censos (INEC).",
          startCol = 2,
          startRow = 24) # cambiar para exportar

writeData(tab_ESED,
          "C1",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2022 - 2025.",
          startCol = 2,
          startRow = 25) # cambiar para exportar

writeData(tab_ESED,
          "C1",
          "Nota 1: Los resultados del año 2022 y 2023  son cifras semi-definitivas, del año 2024 son cifras provisionales, y del año 2025 son cifras preliminares.",
          startCol = 2,
          startRow = 26) # cambiar para exportar

writeData(tab_ESED,
          "C1",
          "Nota 2: Para el I, II y III trimestre 2025, no se dispone de la información del D. M. Quito. La información será actualizada, una vez que se disponga de los datos.",
          startCol = 2,
          startRow = 27) # cambiar para exportar

addStyle(tab_ESED,
         "C1",
         fuente2,
         cols = 2,
         rows = 23:30, # cambiar para exportar
         gridExpand = T)


# Auxiliar para valores con cero (0)

# Definir las combinaciones de columnas y filas en la lista

posiciones <- list(
  c(14, 20), c(14, 21), c(14,22) # Columna 14
)

for (pos in posiciones) {
  writeData(tab_ESED,
            "C1",
            "-",
            startCol = pos[1],  # Columna
            startRow = pos[2],  # Fila
            colNames = FALSE)
}

addStyle(tab_ESED,
         "C1",
         for_dat2,
         cols = 14,
         rows = c(20:22), #cambiar para exportar
         gridExpand = T)


# ============================================================================ #
#######################   Configuración Tabulado C2   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "C2",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\ing_cc2.png",
            height = 3.65,
            width = 40.31,
            startCol = 1,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "C2", 
              rows = 1, 
              heights = "100") 

"Título"
writeData(tab_ESED,
          "C2",
          "Cuadro C2",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "C2",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

"Encabezado"
addStyle(tab_ESED,
         "C2",
         titenc,
         cols = 2:6,
         rows = 4:5,
         gridExpand = T)

"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "C2", 
             cols = 2, 
             widths = "35")

setColWidths(tab_ESED,
             "C2", 
             cols = 3:6, 
             widths = "35")

setColWidths(tab_ESED,
             "C2", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "C2",
              rows = 6:23,
              heights = "40") #cambiar para exportar

setRowHeights(tab_ESED,
              "C2",
              rows = 4:5,
              heights = "30")

"Celdas de títulos"
mergeCells(tab_ESED,
           "C2",
           cols = 2,
           rows = 4:5)

mergeCells(tab_ESED,
           "C2",
           cols = 3,
           rows = 4:5)

mergeCells(tab_ESED,
           "C2",
           cols = 4:6,
           rows = 4)

"Datos y Diseño"
writeData(tab_ESED,
          "C2",
          CC2,
          startCol = 2,
          startRow = 5,
          colNames = T,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "C2",
          "Trimestres",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "C2",
         for_dat1,
         cols = 2,
         rows = c(6:9, 11:14, 16:19, 21:23), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C2",
         for_dat2,
         cols = 3:6,
         rows = c(6:9, 11:14, 16:19, 21:23), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C2",
         for_suma1,
         cols = 2,
         rows = c(10, 15, 20), 
         gridExpand = T)

addStyle(tab_ESED,
         "C2",
         for_suma2,
         cols = 3:6,
         rows = c(10, 15, 20), 
         gridExpand = T)

"Arreglo títulos"
writeData(tab_ESED,
          "C2",
          "Tipo de obra",
          startCol = 4,
          startRow = 4)

writeData(tab_ESED,
          "C2",
          "Total de edificaciones",
          startCol = 3,
          startRow = 4)

"Fuentes y notas"
setRowHeights(tab_ESED, 
              "C2", 
              rows = 24:30,  #cambiar para exportar
              heights = "18")

writeData(tab_ESED,
          "C2",
          "Elaboración: Instituto Nacional de Estadística y Censos (INEC).",
          startCol = 2,
          startRow = 25) # cambiar para exportar

writeData(tab_ESED,
          "C2",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2022 - 2025.",
          startCol = 2,
          startRow = 26) #cambiar para exportar

writeData(tab_ESED,
          "C2",
          "Nota 1: Los resultados del año 2022 y 2023  son cifras semi-definitivas, del año 2024 son cifras provisionales, y del año 2025 son cifras preliminares.",
          startCol = 2,
          startRow = 27) # cambiar para exportar


writeData(tab_ESED,
          "C2",
          "Nota 2: Para el I, II y III trimestre 2025, no se dispone de la información del D. M. Quito. La información será actualizada, una vez que se disponga de los datos.",
          startCol = 2,
          startRow = 28) # cambiar para exportar

addStyle(tab_ESED,
         "C2",
         fuente2,
         cols = 2,
         rows = 24:30, # cambiar para exportar
         gridExpand = T)

# ============================================================================ #
#######################   Configuración Tabulado C3   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "C3",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\ing_cc3.png",
            height = 4.76,
            width = 100.50,
            startCol = 1,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "C3", 
              rows = 1, 
              heights = "130") 

"Título"
writeData(tab_ESED,
          "C3",
          "Cuadro C3",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "C3",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

"Encabezado"
addStyle(tab_ESED,
         "C3",
         titenc,
         cols = 2:18,
         rows = 4,
         gridExpand = T)

"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "C3", 
             cols = 2, 
             widths = "35")

setColWidths(tab_ESED,
             "C3", 
             cols = 3:18, 
             widths = "30")

setColWidths(tab_ESED,
             "C3", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "C3",
              rows = 5:22,
              heights = "40") #cambiar para exportar

setRowHeights(tab_ESED,
              "C3",
              rows = 4,
              heights = "70")

"Datos y Diseño"

writeData(tab_ESED,
          "C3",
          CC3,
          startCol = 2,
          startRow = 4,
          colNames = T,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "C3",
          "Trimestres",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "C3",
         for_dat1,
         cols = 2,
         rows = c(5:8, 10:13, 15:18, 20:22), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C3",
         for_dat2,
         cols = 3:18,
         rows = c(5:8, 10:13, 15:18, 20:22), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C3",
         for_suma1,
         cols = 2,
         rows = c(9, 14, 19 ),
         gridExpand = T)

addStyle(tab_ESED,
         "C3",
         for_suma2,
         cols = 3:18,
         rows = c(9, 14, 19 ),
         gridExpand = T)

"Fuentes y notas"
setRowHeights(tab_ESED, 
              "C3",
              rows = 23:30, # cambiar para exportar
              heights = "18")

writeData(tab_ESED,
          "C3",
          "Elaboración: Instituto Nacional de Estadística y Censos (INEC).",
          startCol = 2,
          startRow = 24) # cambiar para exportar

writeData(tab_ESED,
          "C3",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2022 - 2025.",
          startCol = 2,
          startRow = 25) # cambiar para exportar

writeData(tab_ESED,
          "C3",
          "Nota 1: Los resultados del año 2022 y 2023  son cifras semi-definitivas, del año 2024 son cifras provisionales, y del año 2025 son cifras preliminares.",
          startCol = 2,
          startRow = 26) # cambiar para exportar

writeData(tab_ESED,
          "C3",
          "Nota 2: Para el I, II y III trimestre 2025, no se dispone de la información del D. M. Quito. La información será actualizada, una vez que se disponga de los datos.",
          startCol = 2,
          startRow = 27) # cambiar para exportar

addStyle(tab_ESED,
         "C3",
         fuente2,
         cols = 2,
         rows = 23:30, # cambiar para exportar
         gridExpand = T)


# Auxiliar para valores con cero (0)

# # Definir las combinaciones de columnas y filas en la lista
posiciones <- list(
  c(9, 15),c(9, 20),c(9, 21), c(9,22),  # Columna 9
  c(11, 5), c(11, 6), c(11, 7),c(11, 18),c(11, 20),c(11,22),  # Columna 11
  c(12, 5), c(12, 6), c(12, 10), c(12, 17), c(12,22), # Columna 12
  c(13, 5), # Columna 13
  c(15, 7), c(15, 8),c(15, 10),c(15, 11),c(15, 12),c(15, 13),c(15, 14),c(15, 16),c(15, 17), c(15, 20), c(15,22), # Columna 15
  c(16, 5), c(16, 6), c(16, 7), c(16, 8), c(16, 9),c(16, 10),c(16, 11),c(16, 12),c(16, 13), c(16, 14),c(16, 15),c(16, 17),c(16, 18), c(16, 20), c(16,22) # Columna 16
)

for (pos in posiciones) {
  writeData(tab_ESED,
            "C3",
            "-",
            startCol = pos[1],  # Columna
            startRow = pos[2],  # Fila
            colNames = FALSE)
}

addStyle(tab_ESED,
         "C3",
         for_dat2,
         cols = 3:18,
         rows = c(5:8, 10:13, 15:18, 20:22), # cambiar para exportar
         gridExpand = T)

# ============================================================================ #
#######################   Configuración Tabulado C4   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "C4",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\ing_cc4.png",
            height = 3.65,
            width = 34.62 ,
            startCol = 1,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "C4", 
              rows = 1, 
              heights = "100") 

"Título"
writeData(tab_ESED,
          "C4",
          "Cuadro C4",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "C4",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

"Encabezado"
addStyle(tab_ESED,
         "C4",
         titenc,
         cols = 2:6,
         rows = 4:6,
         gridExpand = T)

"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "C4", 
             cols = 2, 
             widths = "25")

setColWidths(tab_ESED,
             "C4", 
             cols = 3:6, 
             widths = "30")

setColWidths(tab_ESED,
             "C4", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "C4",
              rows = 7:21,
              heights = "40")

setRowHeights(tab_ESED,
              "C4",
              rows = 4:6,
              heights = "30")

"Celdas de títulos"
mergeCells(tab_ESED,
           "C4",
           cols = 2,
           rows = 4:6)

mergeCells(tab_ESED,
           "C4",
           cols = 3:6,
           rows = 4)

mergeCells(tab_ESED,
           "C4",
           cols = 3:4,
           rows = 5)

mergeCells(tab_ESED,
           "C4",
           cols = 5:6,
           rows = 5)

"Datos y Diseño"
writeData(tab_ESED,
          "C4",
          CC4,
          startCol = 2,
          startRow = 7,
          colNames = F,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "C4",
          "Trimestres",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "C4",
         for_dat1,
         cols = 2,
         rows = c(7:10, 12:16, 17:20, 21), 
         gridExpand = T)

addStyle(tab_ESED,
         "C4",
         for_dat2,
         cols = 3:6,
         rows = c(7:10, 12:16, 17:20, 21), 
         gridExpand = T)

addStyle(tab_ESED,
         "C4",
         for_suma1,
         cols = 2,
         rows = c(11, 16, 21 ),
         gridExpand = T)

addStyle(tab_ESED,
         "C4",
         for_suma2,
         cols = 3:6,
         rows = c(11, 16, 21 ),
         gridExpand = T)

"Arreglo títulos"
writeData(tab_ESED,
          "C4",
          "Internet / telefonía celular",
          startCol = 3,
          startRow = 4)

writeData(tab_ESED,
          "C4",
          "Área urbana",
          startCol = 3,
          startRow = 5)

writeData(tab_ESED,
          "C4",
          "Área rural",
          startCol = 5,
          startRow = 5)

writeData(tab_ESED,
          "C4",
          "Con acceso",
          startCol = 3,
          startRow = 6)

writeData(tab_ESED,
          "C4",
          "Sin acceso",
          startCol = 4,
          startRow = 6)

writeData(tab_ESED,
          "C4",
          "Con acceso",
          startCol = 5,
          startRow = 6)

writeData(tab_ESED,
          "C4",
          "Sin acceso",
          startCol = 6,
          startRow = 6)


"Fuentes y notas"

setRowHeights(tab_ESED, 
              "C4" , 
              rows = 22:27,  #cambiar para exportar
              heights = "18")

writeData(tab_ESED,
          "C4",
          "Elaboración: Instituto Nacional de Estadística y Censos (INEC).",
          startCol = 2,
          startRow = 23) # cambiar para exportar

writeData(tab_ESED,
          "C4",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2022 - 2024.",
          startCol = 2,
          startRow = 24) #cambiar para exportar

writeData(tab_ESED,
          "C4",
          "Nota 1: Se excluye la categoría (No responde) para la construcción de este tabulado.",
          startCol = 2,
          startRow = 25) #cambiar para exportar

writeData(tab_ESED,
          "C4",
          "Nota 2: Los resultados del año 2022 y 2023  son cifras semi-definitivas y del año 2024 son cifras provisionales.",
          startCol = 2,
          startRow = 26) # cambiar para exportar

addStyle(tab_ESED,
         "C4",
         fuente2,
         cols = 2,
         rows = 22:27, # cambiar para exportar
         gridExpand = T)


###########################*
# Creación nuevas variables 2025 acceso a internet y celular/convencial
###########################*

#######################   Configuración Tabulado C4 -1   ########################


"Encabezado"
addStyle(tab_ESED,
         "C4",
         titenc,
         cols = 2:6,
         rows = 28:29,
         gridExpand = T)

"Tamaño filas y columnas"

setRowHeights(tab_ESED,
              "C4",
              rows = 30:32,
              heights = "40")#cambiar para exportar

setRowHeights(tab_ESED,
              "C4",
              rows = 28:29,
              heights = "30")

"Celdas de títulos"
mergeCells(tab_ESED,
           "C4",
           cols = 2,
           rows = 28:29)

mergeCells(tab_ESED,
           "C4",
           cols = 3:4,
           rows = 28)

mergeCells(tab_ESED,
           "C4",
           cols = 5:6,
           rows = 28)

"Datos y Diseño"
writeData(tab_ESED,
          "C4",
          CC4_1,
          startCol = 2,
          startRow = 30,
          colNames = F,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "C4",
          "Trimestre",
          startCol = 2,
          startRow = 28)

addStyle(tab_ESED,
         "C4",
         for_dat1,
         cols = 2,
         rows = c(30:32), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C4",
         for_dat2,
         cols = 3:6,
         rows = c(30,31,32), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "C4",
         for_suma1,
         cols = 2,
         rows = c(11, 16, 21 ),
         gridExpand = T)

"Arreglo títulos"
writeData(tab_ESED,
          "C4",
          "Internet",
          startCol = 3,
          startRow = 28)


writeData(tab_ESED,
          "C4",
          "Telefonía celular/convencional",
          startCol = 5,
          startRow = 28)


writeData(tab_ESED,
          "C4",
          "Área urbana",
          startCol = 3,
          startRow = 29)

writeData(tab_ESED,
          "C4",
          "Área rural",
          startCol = 4,
          startRow = 29)


writeData(tab_ESED,
          "C4",
          "Área urbana",
          startCol = 5,
          startRow = 29)

writeData(tab_ESED,
          "C4",
          "Área rural",
          startCol = 6,
          startRow = 29)

"Fuentes y notas"

setRowHeights(tab_ESED, 
              "C4" , 
              rows = 33:38,  #cambiar para exportar
              heights = "18")

writeData(tab_ESED,
          "C4",
          "Elaboración: Instituto Nacional de Estadística y Censos (INEC).",
          startCol = 2,
          startRow = 34) # cambiar para exportar

writeData(tab_ESED,
          "C4",
          "Fuente: Estadísticas de Edificaciones, trimestral 2025.",
          startCol = 2,
          startRow = 35) #cambiar para exportar

writeData(tab_ESED,
          "C4",
          "Nota 1: Se excluye la categoría (No responde) para la construcción de este tabulado.",
          startCol = 2,
          startRow = 36) #cambiar para exportar

writeData(tab_ESED,
          "C4",
          "Nota 2: Los resultados del año 2025 son cifras preliminares.",
          startCol = 2,
          startRow = 37) # cambiar para exportar

writeData(tab_ESED,
          "C4",
          "Nota 3: Para el I, II y III trimestre 2025, no se dispone de la información del D. M. Quito. La información será actualizada, una vez que se disponga de los datos.",
          startCol = 2,
          startRow = 38) # cambiar para exportar

writeData(tab_ESED,
          "C4",
          "Nota 4: La variable de acceso a internet y telefonía celular fue reestructurada en el formulario para el periodo 2025, lo que implicó su recategorización y posterior división en dos variables independientes.",
          startCol = 2,
          startRow = 39) # cambiar para exportar


addStyle(tab_ESED,
         "C4",
         fuente2,
         cols = 2,
         rows = 33:39, # cambiar para exportar
         gridExpand = T)


################################################*

# ============================================================================ #
#######################   Configuración Tabulado D1   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "D1",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\ing_cd1.png",
            height = 4.61,
            width = 70.20,
            startCol = 1,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "D1", 
              rows = 1, 
              heights = "130") 

"Título"
writeData(tab_ESED,
          "D1",
          "Cuadro D1",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "D1",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

setRowHeights(tab_ESED,
              "D1",
              rows = c(4),
              heights = "60")

"Encabezado"
addStyle(tab_ESED,
         "D1",
         titenc,
         cols = 2:18,
         rows = 4,
         gridExpand = T)

"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "D1", 
             cols = 2, 
             widths = "35")

setColWidths(tab_ESED,
             "D1", 
             cols = 3:18, 
             widths = "20")

setColWidths(tab_ESED,
             "D1", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "D1",
              rows = 5:22,
              heights = "40")#cambiar

"Datos y Diseño"
writeData(tab_ESED,
          "D1",
          CD1,
          startCol = 2,
          startRow = 4,
          colNames = T,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "D1",
          "Trimestres / Cantones",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "D1",
         for_dat1,
         cols = 2,
         rows = c(5:8, 10:13, 15:18, 20:22), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "D1",
         for_dat2,
         cols = 3:18,
         rows = c(5:8, 10:13, 15:18, 20:22), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "D1",
         for_suma1,
         cols = 2,
         rows = c(9, 14, 19  ),
         gridExpand = T)

addStyle(tab_ESED,
         "D1",
         for_suma2,
         cols = 3:18,
         rows = c(9, 14, 19  ),
         gridExpand = T)

"Fuentes y notas"
setRowHeights(tab_ESED, 
              "D1",
              rows = 23:30, # cambiar para exportar
              heights = "18")

writeData(tab_ESED,
          "D1",
          "Elaboración: Instituto Nacional de Estadística y Censos (INEC).",
          startCol = 2,
          startRow = 24) # cambiar para exportar

writeData(tab_ESED,
          "D1",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2022 - 2025.",
          startCol = 2,
          startRow = 25) # cambiar para exportar

writeData(tab_ESED,
          "D1",
          "Nota 1: Los resultados del año 2022 y 2023  son cifras semi-definitivas, del año 2024 son cifras provisionales, y del año 2025 son cifras preliminares.",
          startCol = 2,
          startRow = 26) # cambiar para exportar

writeData(tab_ESED,
          "D1",
          "Nota 2: Para el I, II y III trimestre 2025, no se dispone de la información del D. M. Quito. La información será actualizada, una vez que se disponga de los datos.",
          startCol = 2,
          startRow = 27) # cambiar para exportar

addStyle(tab_ESED,
         "D1",
         fuente2,
         cols = 2,
         rows = 23:30, # cambiar para exportar
         gridExpand = T)


# Auxiliar para valores con cero (0)

# Definir las combinaciones de columnas y filas en la lista

posiciones <- list(
  c(14, 20), c(14, 21), c(14,22) # Columna 14
)

for (pos in posiciones) {
  writeData(tab_ESED,
            "D1",
            "-",
            startCol = pos[1],  # Columna
            startRow = pos[2],  # Fila
            colNames = FALSE)
}

addStyle(tab_ESED,
         "D1",
         for_dat2,
         cols = 14,
         rows = c(20:22), #cambiar para exportar
         gridExpand = T)


# ============================================================================ #
#######################   Configuración Tabulado D2   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "D2",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\ing_cd2.png",
            height = 4.15,
            width = 53.71,
            startCol = 1,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "D2", 
              rows = 1, 
              heights = "120") 

"Título"
writeData(tab_ESED,
          "D2",
          "Cuadro D2",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "D2",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

"Encabezado"
addStyle(tab_ESED,
         "D2",
         titenc,
         cols = 2:9,
         rows = 4:5,
         gridExpand = T)

"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "D2", 
             cols = 2, 
             widths = "35")

setColWidths(tab_ESED,
             "D2", 
             cols = 3:9, 
             widths = "30")

setColWidths(tab_ESED,
             "D2", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "D2",
              rows = 6:23,
              heights = "40") #cambiar para exportar

setRowHeights(tab_ESED,
              "D2",
              rows = 4:5,
              heights = "30")

"Celdas de títulos"
mergeCells(tab_ESED,
           "D2",
           cols = 2,
           rows = 4:5)

mergeCells(tab_ESED,
           "D2",
           cols = 3,
           rows = 4:5)

mergeCells(tab_ESED,
           "D2",
           cols = 4:9,
           rows = 4)

"Datos y Diseño"
writeData(tab_ESED,
          "D2",
          CD2,
          startCol = 2,
          startRow = 5,
          colNames = T,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "D2",
          "Trimestres",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "D2",
         for_dat1,
         cols = 2,
         rows = c(6:9, 11:14, 16:19, 21:23), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "D2",
         for_dat2,
         cols = 3:9,
         rows = c(6:9, 11:14, 16:19, 21:23), #cambiar
         gridExpand = T) 

addStyle(tab_ESED,
         "D2",
         for_suma1,
         cols = 2,
         rows = c(10, 15, 20  ), 
         gridExpand = T)

addStyle(tab_ESED,
         "D2",
         for_suma2,
         cols = 3:9,
         rows = c(10, 15, 20 ), 
         gridExpand = T)

"Arreglo títulos"
writeData(tab_ESED,
          "D2",
          "Rangos de área total a construir",
          startCol = 4,
          startRow = 4)

writeData(tab_ESED,
          "D2",
          "Total de viviendas",
          startCol = 3,
          startRow = 4)

"Fuentes y notas"

setRowHeights(tab_ESED, 
              "D2",
              rows = 24:30,  #cambiar para exportar
              heights = "18")

writeData(tab_ESED,
          "D2",
          "Elaboración: Instituto Nacional de Estadística y Censos (INEC).",
          startCol = 2,
          startRow = 25) # cambiar para exportar

writeData(tab_ESED,
          "D2",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2022 - 2025.",
          startCol = 2,
          startRow = 26) #cambiar para exportar

writeData(tab_ESED,
          "D2",
          "Nota 1: Los resultados del año 2022 y 2023  son cifras semi-definitivas, del año 2024 son cifras provisionales, y del año 2025 son cifras preliminares.",
          startCol = 2,
          startRow = 27) # cambiar para exportar

writeData(tab_ESED,
          "D2",
          "Nota 2: Para el I, II y III trimestre 2025, no se dispone de la información del D. M. Quito. La información será actualizada, una vez que se disponga de los datos.",
          startCol = 2,
          startRow = 28) # cambiar para exportar

addStyle(tab_ESED,
         "D2",
         fuente2,
         cols = 2,
         rows = 24:30, # cambiar para exportar
         gridExpand = T)


# ============================================================================ #
#######################   Configuración Tabulado D3   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "D3",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\ing_cd3.png",
            height = 3.65,
            width = 37.70,
            startCol = 1,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "D3", 
              rows = 1, 
              heights = "100") 

"Título"
writeData(tab_ESED,
          "D3",
          "Cuadro D3",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "D3",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

"Encabezado"
addStyle(tab_ESED,
         "D3",
         titenc,
         cols = 2:7,
         rows = 4:5,
         gridExpand = T)

"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "D3", 
             cols = 2, 
             widths = "35")

setColWidths(tab_ESED,
             "D3", 
             cols = 3:7, 
             widths = "25")

setColWidths(tab_ESED,
             "D3", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "D3",
              rows = 6:23,
              heights = "40") # cambiar para exportar

setRowHeights(tab_ESED,
              "D3",
              rows = 4:5,
              heights = "30")

"Celdas de títulos"
mergeCells(tab_ESED,
           "D3",
           cols = 2,
           rows = 4:5)

mergeCells(tab_ESED,
           "D3",
           cols = 3:7,
           rows = 4)

"Datos y Diseño"
writeData(tab_ESED,
          "D3",
          CD3,
          startCol = 2,
          startRow = 5,
          colNames = T,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "D3",
          "Trimestres",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "D3",
         for_dat1,
         cols = 2,
         rows = c(6:9, 11:14, 16:19, 21:23), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "D3",
         for_dat2,
         cols = 3:7,
         rows = c(6:9, 11:14, 16:19, 21:23), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "D3",
         for_suma1,
         cols = 2,
         rows = c(10, 15, 20 ),
         gridExpand = T)

addStyle(tab_ESED,
         "D3",
         for_suma2,
         cols = 3:7,
         rows = c(10, 15, 20), 
         gridExpand = T)

"Arreglo títulos"
writeData(tab_ESED,
          "D3",
          "Número de dormitorios por vivienda",
          startCol = 3,
          startRow = 4)

"Fuentes y notas"

setRowHeights(tab_ESED, 
              "D3",
              rows = 24:30, #cambiar para exportar
              heights = "18")

writeData(tab_ESED,
          "D3",
          "Elaboración: Instituto Nacional de Estadística y Censos (INEC).",
          startCol = 2,
          startRow = 25) # cambiar para exportar

writeData(tab_ESED,
          "D3",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2022 - 2025.",
          startCol = 2,
          startRow = 26) #cambiar para exportar

writeData(tab_ESED,
          "D3",
          "Nota 1: Para la construcción de este tabulado se excluye las reconstrucciones, por tal motivo, el número total de viviendas difiere.",
          startCol = 2,
          startRow = 27) #cambiar para exportar

writeData(tab_ESED,
          "D3",
          "Nota 2: Los resultados del año 2022 y 2023  son cifras semi-definitivas, del año 2024 son cifras provisionales, y del año 2025 son cifras preliminares.",
          startCol = 2,
          startRow = 28) # cambiar para exportar

writeData(tab_ESED,
          "D3",
          "Nota 3: Para el I, II y III trimestre 2025, no se dispone de la información del D. M. Quito. La información será actualizada, una vez que se disponga de los datos.",
          startCol = 2,
          startRow = 29) # cambiar para exportar

addStyle(tab_ESED,
         "D3",
         fuente2,
         cols = 2,
         rows = 24:30, # cambiar para exportar
         gridExpand = T)


# ============================================================================ #
#######################   Configuración Tabulado E1   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "E1",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\ing_ce1.png",
            height = 4.15,
            width = 48.02,
            startCol = 1,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "E1", 
              rows = 1, 
              heights = "165") 


"Título"
writeData(tab_ESED,
          "E1",
          "Cuadro E1",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "E1",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

"Encabezado"
addStyle(tab_ESED,
         "E1",
         titenc,
         cols = 2:8,
         rows = 4:5,
         gridExpand = T)

"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "E1", 
             cols = 2, 
             widths = "35")

setColWidths(tab_ESED,
             "E1", 
             cols = 3:8, 
             widths = "30")

setColWidths(tab_ESED,
             "E1", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "E1",
              rows = 6:23,
              heights = "40")#cambiar para exportar

setRowHeights(tab_ESED,
              "E1",
              rows = 4:5,
              heights = "30")

"Celdas de títulos"
mergeCells(tab_ESED,
           "E1",
           cols = 2,
           rows = 4:5)

mergeCells(tab_ESED,
           "E1",
           cols = 3,
           rows = 4:5)

mergeCells(tab_ESED,
           "E1",
           cols = 4:7,
           rows = 4)

mergeCells(tab_ESED,
           "E1",
           cols = 8,
           rows = 4:5)

"Datos y Diseño"
writeData(tab_ESED,
          "E1",
          CE1,
          startCol = 2,
          startRow = 5,
          colNames = T,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "E1",
          "Trimestres",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "E1",
         for_dat1,
         cols = 2,
         rows = c(6:9, 11:14, 16:19, 21:23),#cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "E1",
         for_dat2,
         cols = 3:8,
         rows = c(6:9, 11:14, 16:19, 21:23), #cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "E1",
         for_suma1,
         cols = 2,
         rows = c(10, 15, 20  ), 
         gridExpand = T)

addStyle(tab_ESED,
         "E1",
         for_suma2,
         cols = 3:8,
         rows = c(10, 15, 20  ), 
         gridExpand = T)

"Arreglo títulos"
writeData(tab_ESED,
          "E1",
          "Área a construir",
          startCol = 4,
          startRow = 4)

writeData(tab_ESED,
          "E1",
          "Superficie total del terreno",
          startCol = 3,
          startRow = 4)

writeData(tab_ESED,
          "E1",
          "Área total para espacios verdes",
          startCol = 8,
          startRow = 4)

"Fuentes y notas"

setRowHeights(tab_ESED, 
              "E1", 
              rows = 24:30, # cambiar para exportar
              heights = "18")

writeData(tab_ESED,
          "E1",
          "Elaboración: Instituto Nacional de Estadística y Censos (INEC).",
          startCol = 2,
          startRow = 25) # cambiar para exportar

writeData(tab_ESED,
          "E1",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2022 - 2025.",
          startCol = 2,
          startRow = 26) # cambiar para exportar

writeData(tab_ESED,
          "E1",
          "* Parqueaderos de las edificaciones con fines no residenciales.",
          startCol = 2,
          startRow = 27) # cambiar para exportar

writeData(tab_ESED,
          "E1",
          "Nota 1: Los resultados del año 2022 y 2023  son cifras semi-definitivas, del año 2024 son cifras provisionales, y del año 2025 son cifras preliminares.",
          startCol = 2,
          startRow = 28) # cambiar para exportar

writeData(tab_ESED,
          "E1",
          "Nota 2: Para el I, II y III trimestre 2025, no se dispone de la información del D. M. Quito. La información será actualizada, una vez que se disponga de los datos.",
          startCol = 2,
          startRow = 29) # cambiar para exportar

addStyle(tab_ESED,
         "E1",
         fuente2,
         cols = 2,
         rows = 24:30, # cambiar para exportar
         gridExpand = T)


# Auxiliar para valores con cero (0)

# Definir las combinaciones de columnas y filas en la lista

posiciones <- list(
  c(7, 8) # Columna 7
)

for (pos in posiciones) {
  writeData(tab_ESED,
            "E1",
            "-",
            startCol = pos[1],  # Columna
            startRow = pos[2],  # Fila
            colNames = FALSE)
}

addStyle(tab_ESED,
         "E1",
         for_dat2,
         cols = 3:8,
         rows = c(6:9), #cambiar para exportar
         gridExpand = T)


# ============================================================================ #
#######################   Configuración Tabulado F1   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "F1",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\ing_cf1.png",
            height = 4.15,
            width = 40.26,
            startCol = 1,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "F1", 
              rows = 1, 
              heights = "120") 

"Título"
writeData(tab_ESED,
          "F1",
          "Cuadro F1",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "F1",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

"Encabezado"
addStyle(tab_ESED,
         "F1",
         titenc,
         cols = 2:6,
         rows = 4:5,
         gridExpand = T)

"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "F1", 
             cols = 2, 
             widths = "35")

setColWidths(tab_ESED,
             "F1", 
             cols = 3:6, 
             widths = "35")

setColWidths(tab_ESED,
             "F1", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "F1",
              rows = 6:23,
              heights = "40")#cambiar para exportar

setRowHeights(tab_ESED,
              "F1",
              rows = 4:5,
              heights = "30")

"Celdas de títulos"
mergeCells(tab_ESED,
           "F1",
           cols = 2,
           rows = 4:5)

mergeCells(tab_ESED,
           "F1",
           cols = 3,
           rows = 4:5)

mergeCells(tab_ESED,
           "F1",
           cols = 4:6,
           rows = 4)

"Datos y Diseño"
writeData(tab_ESED,
          "F1",
          CF1,
          startCol = 2,
          startRow = 5,
          colNames = T,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "F1",
          "Trimestres",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "F1",
         for_dat1,
         cols = 2,
         rows = c(6:9, 11:14, 16:19, 21:23), # cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "F1",
         for_dat2,
         cols = 3:6,
         rows = c(6:9, 11:14, 16:19, 21:23), # cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "F1",
         for_suma1,
         cols = 2,
         rows = c(10, 15, 20 ), 
         gridExpand = T)

addStyle(tab_ESED,
         "F1",
         for_suma2,
         cols = 3:6,
         rows = c(10, 15, 20  ), 
         gridExpand = T)

"Arreglo títulos"
writeData(tab_ESED,
          "F1",
          "Tipo de obra",
          startCol = 4,
          startRow = 4)

writeData(tab_ESED,
          "F1",
          "Monto total de financiamiento",
          startCol = 3,
          startRow = 4)

"Fuentes y notas"

setRowHeights(tab_ESED, 
              "F1",
              rows = 24:30,  # cambiar para exportar
              heights = "18")

writeData(tab_ESED,
          "F1",
          "Elaboración: Instituto Nacional de Estadística y Censos (INEC).",
          startCol = 2,
          startRow = 25) # cambiar para exportar

writeData(tab_ESED,
          "F1",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2022 - 2025.",
          startCol = 2,
          startRow = 26) # cambiar para exportar

writeData(tab_ESED,
          "F1",
          "Nota 1: Los resultados del año 2022 y 2023  son cifras semi-definitivas, del año 2024 son cifras provisionales, y del año 2025 son cifras preliminares.",
          startCol = 2,
          startRow = 27) # cambiar para exportar

writeData(tab_ESED,
          "F1",
          "Nota 2: Para el I, II y III trimestre 2025, no se dispone de la información del D. M. Quito. La información será actualizada, una vez que se disponga de los datos.",
          startCol = 2,
          startRow = 28) # cambiar para exportar

addStyle(tab_ESED,
         "F1",
         fuente2,
         cols = 2,
         rows = 24:30, # cambiar para exportar
         gridExpand = T)

# ============================================================================ #
#######################   Configuración Tabulado F2   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "F2",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\ing_cf2.png",
            height = 4.86,
            width = 100.15,
            startCol = 1,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "F2", 
              rows = 1, 
              heights = "130") 

"Título"
writeData(tab_ESED,
          "F2",
          "Cuadro F2",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "F2",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

"Encabezado"
addStyle(tab_ESED,
         "F2",
         titenc,
         cols = 2:18,
         rows = 4,
         gridExpand = T)

"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "F2", 
             cols = 2, 
             widths = "35")

setColWidths(tab_ESED,
             "F2", 
             cols = 3:18, 
             widths = "30")

setColWidths(tab_ESED,
             "F2", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "F2",
              rows = 5:22,
              heights = "40")#cambiar para exportar

setRowHeights(tab_ESED,
              "F2",
              rows = 4,
              heights = "70")

"Datos y Diseño"
writeData(tab_ESED,
          "F2",
          CF2,
          startCol = 2,
          startRow = 4,
          colNames = T,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "F2",
          "Trimestres",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "F2",
         for_dat1,
         cols = 2,
         rows = c(5:8, 10:13, 15:18, 20:22),  # cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "F2",
         for_dat2,
         cols = 3:18, 
         rows = c(5:8, 10:13, 15:18, 20:22),  # cambiar
         gridExpand = T)

addStyle(tab_ESED,
         "F2",
         for_suma1,
         cols = 2,
         rows = c(9, 14, 19 ),
         gridExpand = T)

addStyle(tab_ESED,
         "F2",
         for_suma2,
         cols = 3:18,
         rows = c(9, 14, 19 ),
         gridExpand = T)

"Fuentes y notas"
setRowHeights(tab_ESED, 
              "F2",
              rows = 23:30, # cambiar para exportar
              heights = "18")

writeData(tab_ESED,
          "F2",
          "Elaboración: Instituto Nacional de Estadística y Censos (INEC).",
          startCol = 2,
          startRow = 24) # cambiar para exportar

writeData(tab_ESED,
          "F2",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2022 - 2025.",
          startCol = 2,
          startRow = 25) # cambiar para exportar

writeData(tab_ESED,
          "F2",
          "Nota 1: Los resultados del año 2022 y 2023  son cifras semi-definitivas, del año 2024 son cifras provisionales, y del año 2025 son cifras preliminares.",
          startCol = 2,
          startRow = 26) # cambiar para exportar

writeData(tab_ESED,
          "F2",
          "Nota 2: Para el I, II y III trimestre 2025, no se dispone de la información del D. M. Quito. La información será actualizada, una vez que se disponga de los datos.",
          startCol = 2,
          startRow = 27) # cambiar para exportar

addStyle(tab_ESED,
         "F2",
         fuente2,
         cols = 2,
         rows = 23:30, # cambiar para exportar
         gridExpand = T)


# Auxiliar para valores con cero (0)

# # Definir las combinaciones de columnas y filas en la lista

posiciones <- list(
  c(9, 15),c(9, 20),c(9, 21), c(9,22), # Columna 9
  c(11, 5), c(11, 6), c(11, 7),c(11, 18),c(11, 20), c(11,22),  # Columna 11
  c(12, 5), c(12, 6), c(12, 10), c(12, 17) ,c(12,22), # Columna 12
  c(13, 5), # Columna 13
  c(15, 7), c(15, 8),c(15, 10),c(15, 11),c(15, 12),c(15, 13),c(15, 14),c(15, 16),c(15, 17), c(15, 20), c(15,22), # Columna 15
  c(16, 5), c(16, 6), c(16, 7), c(16, 8), c(16, 9),c(16, 10),c(16, 11),c(16, 12),c(16, 13), c(16, 14),c(16, 15),c(16, 17),c(16, 18), c(16, 20), c(16,22) # Columna 16
)

for (pos in posiciones) {
  writeData(tab_ESED,
            "F2",
            "-",
            startCol = pos[1],  # Columna
            startRow = pos[2],  # Fila
            colNames = FALSE)
}

addStyle(tab_ESED,
         "F2",
         for_dat2,
         cols = 3:18,
         rows = c(5:8, 10:13, 15:18, 20:22), # cambiar para exportar
         gridExpand = T)

# ============================================================================ #
#######################   Configuración Tabulado F3   ########################
# ============================================================================ #

"Imagen"
insertImage(tab_ESED,
            "F3",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\ing_cf3.png",
            height = 3.93,
            width = 70.20,
            startCol = 1,
            startRow = 1,
            units = "cm")

setRowHeights(tab_ESED, 
              "F3", 
              rows = 1, 
              heights = "120") 

"Título"
writeData(tab_ESED,
          "F3",
          "Cuadro F3",
          startCol = 2,
          startRow = 2)

addStyle(tab_ESED,
         "F3",
         tit_c,
         cols = 2,
         rows = 2,
         gridExpand = T)

setRowHeights(tab_ESED,
              "F3",
              rows = c(4),
              heights = "60")

"Encabezado"
addStyle(tab_ESED,
         "F3",
         titenc,
         cols = 2:18,
         rows = 4,
         gridExpand = T)

"Tamaño filas y columnas"
setColWidths(tab_ESED,
             "F3", 
             cols = 2, 
             widths = "35")

setColWidths(tab_ESED,
             "F3", 
             cols = 3:18, 
             widths = "20")

setColWidths(tab_ESED,
             "F3", 
             cols = 1, 
             widths = "8")

setRowHeights(tab_ESED,
              "F3",
              rows = 5:22,
              heights = "40")#cambiar para exportar

"Datos y Diseño"
writeData(tab_ESED,
          "F3",
          CF3,
          startCol = 2,
          startRow = 4,
          colNames = T,
          rowNames = T,
          withFilter = F)

writeData(tab_ESED,
          "F3",
          "Trimestres / Cantones",
          startCol = 2,
          startRow = 4)

addStyle(tab_ESED,
         "F3",
         for_dat1,
         cols = 2,
         rows = c(5:8, 10:13, 15:18, 20:22),
         gridExpand = T)

addStyle(tab_ESED,
         "F3",
         for_dat2,
         cols = 3:18,
         rows = c(5:8, 10:13, 15:18, 20:22),
         gridExpand = T)

addStyle(tab_ESED,
         "F3",
         for_suma1,
         cols = 2,
         rows = c(9, 14, 19 ),
         gridExpand = T)

addStyle(tab_ESED,
         "F3",
         for_suma2,
         cols = 3:18,
         rows = c(9, 14, 19 ),
         gridExpand = T)

"Fuentes y notas"
setRowHeights(tab_ESED, 
              "F3",
              rows = 23:30, # cambiar para exportar
              heights = "18")

writeData(tab_ESED,
          "F3",
          "Elaboración: Instituto Nacional de Estadística y Censos (INEC).",
          startCol = 2,
          startRow = 24) # cambiar para exportar

writeData(tab_ESED,
          "F3",
          "Fuente: Estadísticas de Edificaciones, acumulada trimestral 2022 - 2025.",
          startCol = 2,
          startRow = 25) # cambiar para exportar

writeData(tab_ESED,
          "F3",
          "Nota 1: Los resultados del año 2022 y 2023  son cifras semi-definitivas, del año 2024 son cifras provisionales, y del año 2025 son cifras preliminares.",
          startCol = 2,
          startRow = 26) # cambiar para exportar

writeData(tab_ESED,
          "F3",
          "Nota 2: Para el I, II y III trimestre 2025, no se dispone de la información del D. M. Quito. La información será actualizada, una vez que se disponga de los datos.",
          startCol = 2,
          startRow = 27) # cambiar para exportar

addStyle(tab_ESED,
         "F3",
         fuente2,
         cols = 2,
         rows = 23:30, # cambiar para exportar
         gridExpand = T)


# Auxiliar para valores con cero (0)

# Definir las combinaciones de columnas y filas en la lista

posiciones <- list(
  c(14, 20), c(14, 21), c(14,22)# Columna 14
)

for (pos in posiciones) {
  writeData(tab_ESED,
            "F3",
            "-",
            startCol = pos[1],  # Columna
            startRow = pos[2],  # Fila
            colNames = FALSE)
}

addStyle(tab_ESED,
         "F3",
         for_dat2,
         cols = 14,
         rows = c(20:22), #cambiar para exportar
         gridExpand = T)

# ============================================================================ #
##################   Creación de hipervínculos en el Índice   ##################
# ============================================================================ #

"A1"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 6,
             x = makeHyperlinkString(
               sheet = "A1",
               row = 1,
               col = 1,
               text = "Resultados de las principales variables"))

"B1"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 8,
             x = makeHyperlinkString(
               sheet = "B1",
               row = 1,
               col = 1,
               text = "Permisos de construcción, por trimestres, según cantones"))

"C1"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 10,
             x = makeHyperlinkString(
               sheet = "C1",
               row = 1,
               col = 1,
               text = "Edificaciones a construir, por trimestres, según cantones"))

"C2"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 11,
             x = makeHyperlinkString(
               sheet = "C2",
               row = 1,
               col = 1,
               text = "Edificaciones a construir, por trimestres, según tipo de obra"))

"C3"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 12,
             x = makeHyperlinkString(
               sheet = "C3",
               row = 1,
               col = 1,
               text = "Edificaciones a construir, por trimestres, según uso de la edificación"))

"C4"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 13,
             x = makeHyperlinkString(
               sheet = "C4",
               row = 1,
               col = 1,
               text = "Edificaciones a construir, por trimestres, según acceso a internet – telefonía celular/convencional"))

"D1"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 15,
             x = makeHyperlinkString(
               sheet = "D1",
               row = 1,
               col = 1,
               text = "Viviendas a construir, por trimestres, según cantones"))

"D2"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 16,
             x = makeHyperlinkString(
               sheet = "D2",
               row = 1,
               col = 1,
               text = "Viviendas a construir, por trimestres, según metros cuadrados"))

"D3"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 17,
             x = makeHyperlinkString(
               sheet = "D3",
               row = 1,
               col = 1,
               text = "Viviendas a construir, por trimestres, según número de dormitorios"))

"E1"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 19,
             x = makeHyperlinkString(
               sheet = "E1",
               row = 1,
               col = 1,
               text = "Superficie del terreno y área a construir, por trimestres, según uso de la edificación"))

"F1"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 21,
             x = makeHyperlinkString(
               sheet = "F1",
               row = 1,
               col = 1,
               text = "Valor estimado del financiamiento de las edificaciones, por trimestres, según tipo de obra"))

"F2"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 22,
             x = makeHyperlinkString(
               sheet = "F2",
               row = 1,
               col = 1,
               text = "Valor estimado del financiamiento de las edificaciones, por trimestres, según uso de la edificación"))

"F3"
writeFormula(tab_ESED,
             "ÍNDICE",
             startCol = 3,
             startRow = 23,
             x = makeHyperlinkString(
               sheet = "F3",
               row = 1,
               col = 1,
               text = "Valor estimado de las edificaciones, por trimestres, según cantones"))


"Reaplicación de diseño"
addStyle(tab_ESED,
         "ÍNDICE",
         fill_a1,
         cols = 3,
         rows = c(6, 8, 10:13, 15:17, 19, 21:23),
         gridExpand = T)


# ============================================================================ #
##################   Creación de hipervínculos en cada hoja   ##################
# ============================================================================ #

"Diseño de la celda del hipervínculo"
hyp_ind <- createStyle(fontName = "Century Gothic",
                       fontSize = 15,
                       fontColour = "#FFFFFF",
                       textDecoration = "bold",
                       fgFill = "#7F7F7F",
                       halign = "center",
                       valign = "center",
                       wrapText = T)

"A1"
insertImage(tab_ESED,
            "A1",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\flecha.png",
            height = 2.15,
            width = 2.12,
            startCol = 9,
            startRow = 4,
            units = "cm")

writeFormula(tab_ESED,
             "A1",
             startCol = 10,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "A1",
         hyp_ind,
         cols = 10,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "A1",
             cols = 10,
             widths = "15")

"B1"
insertImage(tab_ESED,
            "B1",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\flecha.png",
            height = 2.15,
            width = 2.12,
            startCol = 19,
            startRow = 4,
            units = "cm")

writeFormula(tab_ESED,
             "B1",
             startCol = 20,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "B1",
         hyp_ind,
         cols = 20,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "B1",
             cols = 20,
             widths = "15")

"C1"
insertImage(tab_ESED,
            "C1",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\flecha.png",
            height = 2.15,
            width = 2.12,
            startCol = 19,
            startRow = 4,
            units = "cm")

writeFormula(tab_ESED,
             "C1",
             startCol = 20,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "C1",
         hyp_ind,
         cols = 20,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "C1",
             cols = 20,
             widths = "15")

"C2"
insertImage(tab_ESED,
            "C2",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\flecha.png",
            height = 2.15,
            width = 2.12,
            startCol = 7,
            startRow = 4,
            units = "cm")

mergeCells(tab_ESED,
           "C2",
           cols = 8,
           rows = 4:5)

writeFormula(tab_ESED,
             "C2",
             startCol = 8,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "C2",
         hyp_ind,
         cols = 8,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "C2",
             cols = 8,
             widths = "15")

"C3"
insertImage(tab_ESED,
            "C3",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\flecha.png",
            height = 2.56,
            width = 2.12,
            startCol = 19,
            startRow = 4,
            units = "cm")

writeFormula(tab_ESED,
             "C3",
             startCol = 20,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "C3",
         hyp_ind,
         cols = 20,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "C3",
             cols = 20,
             widths = "15")

"C4"
insertImage(tab_ESED,
            "C4",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\flecha.png",
            height = 2.15,
            width = 2.12,
            startCol = 7,
            startRow = 4,
            units = "cm")

mergeCells(tab_ESED,
           "C4",
           cols = 8,
           rows = 4:5)

writeFormula(tab_ESED,
             "C4",
             startCol = 8,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "C4",
         hyp_ind,
         cols = 8,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "C4",
             cols = 8,
             widths = "15")

"D1"
insertImage(tab_ESED,
            "D1",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\flecha.png",
            height = 2.15,
            width = 2.12,
            startCol = 19,
            startRow = 4,
            units = "cm")

writeFormula(tab_ESED,
             "D1",
             startCol = 20,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "D1",
         hyp_ind,
         cols = 20,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "D1",
             cols = 20,
             widths = "15")

"D2"
insertImage(tab_ESED,
            "D2",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\flecha.png",
            height = 2.15,
            width = 2.12,
            startCol = 10,
            startRow = 4,
            units = "cm")

mergeCells(tab_ESED,
           "D2",
           cols = 11,
           rows = 4:5)

writeFormula(tab_ESED,
             "D2",
             startCol = 11,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "D2",
         hyp_ind,
         cols = 11,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "D2",
             cols = 11,
             widths = "15")

"D3"
insertImage(tab_ESED,
            "D3",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\flecha.png",
            height = 2.15,
            width = 2.12,
            startCol = 8,
            startRow = 4,
            units = "cm")

mergeCells(tab_ESED,
           "D3",
           cols = 9,
           rows = 4:5)

writeFormula(tab_ESED,
             "D3",
             startCol = 9,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "D3",
         hyp_ind,
         cols = 9,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "D3",
             cols = 9,
             widths = "15")

"E1"
insertImage(tab_ESED,
            "E1",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\flecha.png",
            height = 2.15,
            width = 2.12,
            startCol = 9,
            startRow = 4,
            units = "cm")

mergeCells(tab_ESED,
           "E1",
           cols = 10,
           rows = 4:5)

writeFormula(tab_ESED,
             "E1",
             startCol = 10,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "E1",
         hyp_ind,
         cols = 10,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "E1",
             cols = 10,
             widths = "15")

"F1"
insertImage(tab_ESED,
            "F1",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\flecha.png",
            height = 2.15,
            width = 2.12,
            startCol = 7,
            startRow = 4,
            units = "cm")

mergeCells(tab_ESED,
           "F1",
           cols = 8,
           rows = 4:5)

writeFormula(tab_ESED,
             "F1",
             startCol = 8,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "F1",
         hyp_ind,
         cols = 8,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "F1",
             cols = 8,
             widths = "15")

"F2"
insertImage(tab_ESED,
            "F2",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\flecha.png",
            height = 2.56,
            width = 2.12,
            startCol = 19,
            startRow = 4,
            units = "cm")

writeFormula(tab_ESED,
             "F2",
             startCol = 20,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "F2",
         hyp_ind,
         cols = 20,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "F2",
             cols = 20,
             widths = "15")

"F3"
insertImage(tab_ESED,
            "F3",
            "C:\\Tabulados ESED IIIT 2025\\Títulos tabulados\\flecha.png",
            height = 2.15,
            width = 2.12,
            startCol = 19,
            startRow = 4,
            units = "cm")

writeFormula(tab_ESED,
             "F3",
             startCol = 20,
             startRow = 4,
             x = makeHyperlinkString(
               sheet = "ÍNDICE",
               col = 1,
               row = 1,
               text = "Índice"))

addStyle(tab_ESED,
         "F3",
         hyp_ind,
         cols = 20,
         rows = 4,
         gridExpand = T)

setColWidths(tab_ESED,
             "F3",
             cols = 20,
             widths = "15")

# ============================================================================ #
####################   Mejoras finales al diseño del Excel   ###################
# ============================================================================ #

showGridLines(tab_ESED, "ÍNDICE", showGridLines = F)
showGridLines(tab_ESED, "A1", showGridLines = F)
showGridLines(tab_ESED, "B1", showGridLines = F)
showGridLines(tab_ESED, "C1", showGridLines = F)
showGridLines(tab_ESED, "C2", showGridLines = F)
showGridLines(tab_ESED, "C3", showGridLines = F)
showGridLines(tab_ESED, "C4", showGridLines = F)
showGridLines(tab_ESED, "D1", showGridLines = F)
showGridLines(tab_ESED, "D2", showGridLines = F)
showGridLines(tab_ESED, "D3", showGridLines = F)
showGridLines(tab_ESED, "E1", showGridLines = F)
showGridLines(tab_ESED, "F1", showGridLines = F)
showGridLines(tab_ESED, "F2", showGridLines = F)
showGridLines(tab_ESED, "F3", showGridLines = F)

# ============================================================================ #
#####      Eliminación de variables auxiliares y otros componentes      ########
# ============================================================================ #

# # Eliminación de variables auxiliares 
# rm(auxqcb1 , auxqcc1 , auxqcd1, auxqcf3)

# Eliminación de otros componentes 

rm(fill_a1 , fill_abcd , fill_rest1 , fill_rest2 , foot1 ,foot2 , for_dat1,
   for_dat2 , for_suma1 , for_suma2 ,fuente, fuente2 , hyp_ind , tit_c , titenc , c1,c2, ind)

# ============================================================================ #
#####################   Exportación de tabulados a Excel   #####################
# ============================================================================ #

saveWorkbook(tab_ESED, "5. 2025_ESED_Tabulados-IIIT.xlsx", overwrite = T)

#####                           FIN SCRIPT                             #####